
import React, { useEffect, useState } from 'react';
import { useData } from '../../context/DataContext';

export const WaitingRoom: React.FC = () => {
  const { appointments } = useData();
  const [currentTime, setCurrentTime] = useState(new Date());

  // Update clock
  useEffect(() => {
    const timer = setInterval(() => setCurrentTime(new Date()), 1000);
    return () => clearInterval(timer);
  }, []);

  // Filter today's active appointments
  const todayStr = new Date().toISOString().split('T')[0];
  const activeAppointments = appointments
    .filter(a => a.date === todayStr && (a.status === 'confirmed' || a.status === 'pending'))
    .sort((a, b) => a.time.localeCompare(b.time));

  const currentPatient = activeAppointments[0];
  const nextPatients = activeAppointments.slice(1, 5);

  return (
    <div className="min-h-screen bg-gray-900 text-white font-sans overflow-hidden flex flex-col">
      {/* Header */}
      <header className="bg-gray-800 p-6 flex justify-between items-center shadow-lg border-b border-gray-700">
          <div className="flex items-center gap-4">
              <div className="w-16 h-16 bg-blue-600 rounded-xl flex items-center justify-center text-3xl">
                  <i className="fas fa-clinic-medical"></i>
              </div>
              <div>
                  <h1 className="text-3xl font-bold">عيادة الشفاء</h1>
                  <p className="text-gray-400">شاشة الانتظار</p>
              </div>
          </div>
          <div className="text-right">
              <div className="text-4xl font-mono font-bold tracking-widest text-blue-400">
                  {currentTime.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' })}
              </div>
              <div className="text-gray-400 text-sm mt-1">
                  {currentTime.toLocaleDateString('ar-EG', { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' })}
              </div>
          </div>
      </header>

      {/* Main Content */}
      <main className="flex-1 p-8 grid grid-cols-2 gap-8">
          
          {/* Current Patient (Left) */}
          <div className="bg-gray-800 rounded-3xl p-10 flex flex-col items-center justify-center text-center border-l-8 border-green-500 shadow-2xl relative overflow-hidden">
              <div className="absolute top-0 left-0 w-full h-full bg-green-500/5 z-0 animate-pulse"></div>
              <div className="relative z-10">
                  <h2 className="text-3xl text-gray-400 mb-8 uppercase tracking-widest font-semibold">الدور الحالي</h2>
                  {currentPatient ? (
                      <>
                        <div className="text-7xl md:text-8xl font-bold mb-6 text-white leading-tight">
                            {currentPatient.patientName}
                        </div>
                        <div className="inline-block bg-gray-700 px-8 py-3 rounded-full text-2xl text-green-400 font-mono">
                            غرفة الكشف: 1
                        </div>
                      </>
                  ) : (
                      <p className="text-4xl text-gray-500">لا يوجد مرضى في الانتظار حالياً</p>
                  )}
              </div>
          </div>

          {/* Next Patients (Right) */}
          <div className="bg-gray-800 rounded-3xl p-8 border border-gray-700 flex flex-col">
              <h3 className="text-2xl text-gray-400 mb-6 flex items-center gap-3">
                  <i className="fas fa-users"></i>
                  قائمة الانتظار
              </h3>
              <div className="flex-1 space-y-4">
                  {nextPatients.map((patient, idx) => (
                      <div key={patient.id} className="bg-gray-700/50 p-6 rounded-2xl flex justify-between items-center border-r-4 border-blue-500">
                          <div className="flex items-center gap-4">
                              <span className="w-10 h-10 rounded-full bg-gray-600 flex items-center justify-center font-bold text-gray-300">
                                  {idx + 1}
                              </span>
                              <span className="text-2xl font-bold text-gray-200">{patient.patientName}</span>
                          </div>
                          <span className="text-xl font-mono text-blue-300">{patient.time}</span>
                      </div>
                  ))}
                  {nextPatients.length === 0 && (
                      <div className="h-full flex items-center justify-center text-gray-600">
                          القائمة فارغة
                      </div>
                  )}
              </div>
          </div>
      </main>

      {/* Ticker / Footer */}
      <footer className="bg-blue-900 py-3 overflow-hidden whitespace-nowrap">
          <div className="animate-marquee inline-block text-lg font-medium text-blue-100">
              يرجى تجهيز بطاقة الهوية عند الاستقبال • نتمنى لكم الشفاء العاجل • أوقات الدوام من 9 ص إلى 9 م
          </div>
      </footer>
      
      <style>{`
        @keyframes marquee {
            0% { transform: translateX(100%); }
            100% { transform: translateX(-100%); }
        }
        .animate-marquee {
            animation: marquee 20s linear infinite;
        }
      `}</style>
    </div>
  );
};
