
import React, { useState, useEffect } from 'react';
import { useData } from '../../context/DataContext';
import { SiteContent } from '../../types';

export const SiteEditor: React.FC = () => {
  const { tenants, clinicType, updateTenantSiteContent } = useData();
  
  // Find current tenant (Simplified for demo)
  const currentTenant = tenants.find(t => t.clinicType === clinicType);
  
  const [content, setContent] = useState<SiteContent>({
      heroTitle: 'رعايتكم الصحية أولويتنا القصوى',
      heroSubtitle: 'نقدم أفضل الخدمات الطبية بأحدث التقنيات لضمان صحتكم وراحتكم.',
      aboutTitle: 'من نحن؟',
      aboutText: 'نلتزم بتقديم أعلى معايير الرعاية الصحية لمرضانا. عيادتنا مجهزة بأحدث التقنيات وفريق طبي ذو خبرة عالية.',
      contactAddress: 'العنوان المسجل في الإعدادات',
      contactPhone: currentTenant?.phone || '',
      contactEmail: currentTenant?.email || '',
  });

  useEffect(() => {
      if (currentTenant?.siteContent) {
          setContent(currentTenant.siteContent);
      }
  }, [currentTenant]);

  const handleChange = (field: keyof SiteContent, value: string) => {
      setContent(prev => ({ ...prev, [field]: value }));
  };

  const handleSave = () => {
      updateTenantSiteContent(content);
  };

  return (
    <div className="space-y-6">
      <div className="flex justify-between items-center mb-6">
        <div>
          <h2 className="text-2xl font-bold text-gray-800">محرر الموقع (CMS)</h2>
          <p className="text-gray-500">تعديل محتوى الصفحة الرئيسية للموقع العام</p>
        </div>
        <div className="flex gap-2">
            <a 
                href={`/#/site/${currentTenant?.domain || 'ahmed.clinicpro.com'}`} 
                target="_blank" 
                rel="noreferrer"
                className="bg-white border border-gray-300 text-gray-700 px-4 py-2.5 rounded-xl shadow-sm transition-colors hover:bg-gray-50 flex items-center gap-2"
            >
                <i className="fas fa-external-link-alt"></i>
                معاينة الموقع
            </a>
            <button 
            onClick={handleSave}
            className="bg-primary hover:bg-primary-dark text-white px-5 py-2.5 rounded-xl shadow-md transition-colors flex items-center gap-2"
            >
            <i className="fas fa-save"></i>
            <span>حفظ التغييرات</span>
            </button>
        </div>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
          {/* Hero Section Editor */}
          <div className="bg-white rounded-2xl shadow-sm border border-gray-100 p-6">
              <h3 className="font-bold text-gray-800 mb-4 flex items-center gap-2">
                  <i className="fas fa-home text-primary"></i>
                  واجهة الاستقبال (Hero Section)
              </h3>
              <div className="space-y-4">
                  <div>
                      <label className="block text-sm font-medium text-gray-700 mb-1">العنوان الرئيسي (Hero Title)</label>
                      <input 
                        type="text" 
                        value={content.heroTitle}
                        onChange={e => handleChange('heroTitle', e.target.value)}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-primary focus:border-primary bg-white text-gray-900"
                      />
                  </div>
                  <div>
                      <label className="block text-sm font-medium text-gray-700 mb-1">النص الترحيبي (Subtitle)</label>
                      <textarea 
                        rows={3}
                        value={content.heroSubtitle}
                        onChange={e => handleChange('heroSubtitle', e.target.value)}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-primary focus:border-primary bg-white text-gray-900"
                      ></textarea>
                  </div>
              </div>
          </div>

          {/* Contact Info Editor */}
          <div className="bg-white rounded-2xl shadow-sm border border-gray-100 p-6">
              <h3 className="font-bold text-gray-800 mb-4 flex items-center gap-2">
                  <i className="fas fa-address-card text-blue-500"></i>
                  معلومات التواصل (Footer)
              </h3>
              <div className="space-y-4">
                  <div>
                      <label className="block text-sm font-medium text-gray-700 mb-1">رقم الهاتف الظاهر</label>
                      <input 
                        type="text" 
                        value={content.contactPhone}
                        onChange={e => handleChange('contactPhone', e.target.value)}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-primary focus:border-primary bg-white text-gray-900"
                        dir="ltr"
                      />
                  </div>
                  <div>
                      <label className="block text-sm font-medium text-gray-700 mb-1">البريد الإلكتروني</label>
                      <input 
                        type="text" 
                        value={content.contactEmail}
                        onChange={e => handleChange('contactEmail', e.target.value)}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-primary focus:border-primary bg-white text-gray-900"
                        dir="ltr"
                      />
                  </div>
                  <div>
                      <label className="block text-sm font-medium text-gray-700 mb-1">العنوان النصي</label>
                      <input 
                        type="text" 
                        value={content.contactAddress}
                        onChange={e => handleChange('contactAddress', e.target.value)}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-primary focus:border-primary bg-white text-gray-900"
                      />
                  </div>
              </div>
          </div>

          {/* About Us Editor */}
          <div className="bg-white rounded-2xl shadow-sm border border-gray-100 p-6 lg:col-span-2">
              <h3 className="font-bold text-gray-800 mb-4 flex items-center gap-2">
                  <i className="fas fa-info-circle text-green-500"></i>
                  قسم من نحن (About Us)
              </h3>
              <div className="space-y-4">
                  <div>
                      <label className="block text-sm font-medium text-gray-700 mb-1">عنوان القسم</label>
                      <input 
                        type="text" 
                        value={content.aboutTitle}
                        onChange={e => handleChange('aboutTitle', e.target.value)}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-primary focus:border-primary bg-white text-gray-900"
                      />
                  </div>
                  <div>
                      <label className="block text-sm font-medium text-gray-700 mb-1">نص الوصف</label>
                      <textarea 
                        rows={4}
                        value={content.aboutText}
                        onChange={e => handleChange('aboutText', e.target.value)}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-primary focus:border-primary bg-white text-gray-900"
                      ></textarea>
                  </div>
              </div>
          </div>
      </div>
    </div>
  );
};
