
import React, { useState } from 'react';
import { useData } from '../../context/DataContext';
import { PrescriptionModal } from '../../components/specialties/PrescriptionModal';

export const Prescriptions: React.FC = () => {
  const { prescriptions } = useData();
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [searchTerm, setSearchTerm] = useState('');

  const filtered = prescriptions.filter(p => p.patientName.includes(searchTerm) || p.id.includes(searchTerm));

  const handlePrint = (id: string) => {
      // In a real app, generate PDF
      window.print();
  };

  return (
    <div className="space-y-6">
      <div className="flex justify-between items-center mb-6">
        <div>
          <h2 className="text-2xl font-bold text-gray-800">الوصفات الطبية</h2>
          <p className="text-gray-500">سجل الوصفات الإلكترونية الصادرة</p>
        </div>
        <button 
          onClick={() => setIsModalOpen(true)}
          className="bg-primary hover:bg-primary-dark text-white px-5 py-2.5 rounded-xl shadow-md transition-colors flex items-center gap-2"
        >
          <i className="fas fa-file-prescription"></i>
          <span>وصفة جديدة</span>
        </button>
      </div>

      <div className="bg-white rounded-2xl shadow-sm border border-gray-100 p-4 mb-6">
          <div className="relative">
            <input 
              type="text" 
              placeholder="بحث باسم المريض أو رقم الوصفة..." 
              className="w-full pl-4 pr-10 py-2 border border-gray-200 rounded-lg focus:outline-none focus:border-primary bg-white text-gray-900"
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
            />
            <i className="fas fa-search absolute left-3 top-3 text-gray-400"></i>
          </div>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          {filtered.map(p => (
              <div key={p.id} className="bg-white p-5 rounded-2xl shadow-sm border border-gray-100 hover:shadow-md transition-shadow">
                  <div className="flex justify-between items-start mb-3">
                      <div>
                          <h4 className="font-bold text-gray-800">{p.patientName}</h4>
                          <span className="text-xs text-gray-500 bg-gray-100 px-2 py-0.5 rounded">{p.id}</span>
                      </div>
                      <span className="text-sm text-gray-500">{p.date}</span>
                  </div>
                  <div className="space-y-2 mb-4 border-t border-gray-50 pt-3">
                      {p.items.slice(0, 2).map((item, idx) => (
                          <div key={idx} className="flex justify-between text-sm">
                              <span className="font-medium text-gray-700">• {item.drugName}</span>
                              <span className="text-gray-500">{item.dosage}</span>
                          </div>
                      ))}
                      {p.items.length > 2 && <p className="text-xs text-primary font-semibold">+ {p.items.length - 2} أدوية أخرى</p>}
                  </div>
                  <button 
                    onClick={() => handlePrint(p.id)}
                    className="w-full py-2 border border-primary text-primary rounded-lg text-sm font-bold hover:bg-primary hover:text-white transition-colors flex items-center justify-center gap-2"
                  >
                      <i className="fas fa-print"></i> طباعة الوصفة
                  </button>
              </div>
          ))}
          {filtered.length === 0 && (
              <div className="col-span-full text-center py-12 text-gray-400">لا توجد وصفات طبية</div>
          )}
      </div>

      <PrescriptionModal isOpen={isModalOpen} onClose={() => setIsModalOpen(false)} />
    </div>
  );
};
