
import React, { useState } from 'react';
import { useToast } from '../../context/ToastContext';

interface Role {
    id: string;
    name: string;
    permissions: string[];
}

export const RoleManager: React.FC = () => {
    const { addToast } = useToast();
    
    // Mock Roles
    const [roles, setRoles] = useState<Role[]>([
        { id: '1', name: 'طبيب (Doctor)', permissions: ['view_patients', 'edit_patients', 'create_prescriptions'] },
        { id: '2', name: 'استقبال (Reception)', permissions: ['view_patients', 'create_appointments'] },
        { id: '3', name: 'ممرض (Nurse)', permissions: ['view_patients', 'add_vitals'] },
    ]);

    const allPermissions = [
        { id: 'view_patients', label: 'عرض سجلات المرضى' },
        { id: 'edit_patients', label: 'تعديل البيانات الطبية' },
        { id: 'delete_patients', label: 'حذف المرضى' },
        { id: 'create_appointments', label: 'حجز المواعيد' },
        { id: 'view_financials', label: 'الاطلاع على الحسابات' },
        { id: 'manage_settings', label: 'إدارة الإعدادات' },
    ];

    const togglePermission = (roleId: string, permId: string) => {
        setRoles(prev => prev.map(role => {
            if (role.id === roleId) {
                const hasPerm = role.permissions.includes(permId);
                return {
                    ...role,
                    permissions: hasPerm 
                        ? role.permissions.filter(p => p !== permId)
                        : [...role.permissions, permId]
                };
            }
            return role;
        }));
        addToast('تم تحديث الصلاحيات', 'info');
    };

    return (
        <div className="space-y-6">
            <header className="mb-6">
                <h2 className="text-2xl font-bold text-gray-800">إدارة الأدوار والصلاحيات</h2>
                <p className="text-gray-500">تخصيص صلاحيات الوصول لكل دور وظيفي في العيادة</p>
            </header>

            <div className="grid gap-6">
                {roles.map(role => (
                    <div key={role.id} className="bg-white rounded-2xl shadow-sm border border-gray-100 p-6">
                        <div className="flex justify-between items-center mb-4 border-b border-gray-100 pb-2">
                            <h3 className="font-bold text-lg text-gray-800">{role.name}</h3>
                            <span className="bg-blue-50 text-blue-600 text-xs px-2 py-1 rounded font-bold">{role.permissions.length} صلاحيات</span>
                        </div>
                        
                        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-3">
                            {allPermissions.map(perm => (
                                <label key={perm.id} className="flex items-center gap-2 cursor-pointer p-2 rounded-lg hover:bg-gray-50 border border-transparent hover:border-gray-200 transition-all">
                                    <input 
                                        type="checkbox" 
                                        checked={role.permissions.includes(perm.id)}
                                        onChange={() => togglePermission(role.id, perm.id)}
                                        className="w-4 h-4 text-primary rounded focus:ring-primary"
                                    />
                                    <span className="text-sm text-gray-700">{perm.label}</span>
                                </label>
                            ))}
                        </div>
                    </div>
                ))}
            </div>
            
            <div className="flex justify-end">
                <button className="bg-primary text-white px-6 py-2 rounded-xl font-bold shadow hover:bg-primary-dark transition-colors">
                    <i className="fas fa-save ml-2"></i> حفظ التغييرات
                </button>
            </div>
        </div>
    );
};
