
import React, { useState, useEffect } from 'react';
import { useLocation } from 'react-router-dom';
import { useData } from '../../context/DataContext';
import { useToast } from '../../context/ToastContext';
import { FormField, ClinicType, CustomForm, FieldType } from '../../types';
import { Modal } from '../../components/Modal';

export const FormBuilder: React.FC = () => {
  const { customForms, addCustomForm, deleteCustomForm, clinicType } = useData();
  const { addToast } = useToast();
  const location = useLocation();
  
  // Check if we are in doctor mode
  const isDoctorMode = location.pathname.includes('/doctor');

  const [isModalOpen, setIsModalOpen] = useState(false);
  const [isPreviewOpen, setIsPreviewOpen] = useState(false);
  
  // Builder State
  const [formTitle, setFormTitle] = useState('');
  const [formDesc, setFormDesc] = useState('');
  const [targetLocation, setTargetLocation] = useState('menu'); // New: Location state
  const [targetClinic, setTargetClinic] = useState<ClinicType | 'all'>('all');
  const [fields, setFields] = useState<FormField[]>([]);
  
  // Field Editor State
  const [selectedFieldId, setSelectedFieldId] = useState<string | null>(null);

  // Preview State (Wizard)
  const [previewStep, setPreviewStep] = useState(0);

  // Set default clinic for doctor
  useEffect(() => {
      if (isDoctorMode) {
          setTargetClinic(clinicType);
      }
  }, [isDoctorMode, clinicType]);

  const fieldTypes: { type: FieldType; label: string; icon: string; category: string }[] = [
      // Data Collection
      { type: 'text', label: 'اسم / نص', icon: 'fa-font', category: 'data' },
      { type: 'email', label: 'بريد إلكتروني', icon: 'fa-envelope', category: 'data' },
      { type: 'phone_input', label: 'رقم هاتف', icon: 'fa-phone', category: 'data' },
      { type: 'address', label: 'العنوان', icon: 'fa-map-marker-alt', category: 'data' },
      { type: 'date', label: 'تاريخ', icon: 'fa-calendar', category: 'data' },
      { type: 'time', label: 'وقت', icon: 'fa-clock', category: 'data' },
      
      // Selection
      { type: 'select', label: 'قائمة منسدلة', icon: 'fa-list-ul', category: 'inputs' },
      { type: 'checkbox', label: 'خانة اختيار', icon: 'fa-check-square', category: 'inputs' },
      { type: 'radio', label: 'خيارات (Radio)', icon: 'fa-dot-circle', category: 'inputs' },
      { type: 'multiselect', label: 'تحديد متعدد', icon: 'fa-tags', category: 'inputs' },
      { type: 'switch', label: 'مفتاح (Switch)', icon: 'fa-toggle-on', category: 'inputs' },
      { type: 'rating', label: 'تقييم نجوم', icon: 'fa-star', category: 'inputs' },
      
      // Media & Interactive (Expanded)
      { type: 'static_image', label: 'صورة ثابتة', icon: 'fa-image', category: 'media' },
      { type: 'video', label: 'فيديو (YouTube)', icon: 'fa-video', category: 'media' },
      { type: 'map', label: 'خريطة جوجل', icon: 'fa-map', category: 'media' },
      { type: 'chart', label: 'رسم بياني', icon: 'fa-chart-pie', category: 'media' },
      { type: 'html_block', label: 'كود HTML/تضمين', icon: 'fa-code', category: 'media' },
      { type: 'social_share', label: 'مشاركة اجتماعية', icon: 'fa-share-alt', category: 'media' },
      
      // Layout & Widgets (Expanded)
      { type: 'header', label: 'عنوان قسم', icon: 'fa-heading', category: 'layout' },
      { type: 'paragraph', label: 'فقرة نصية', icon: 'fa-paragraph', category: 'layout' },
      { type: 'divider', label: 'فاصل أفقي', icon: 'fa-minus', category: 'layout' },
      { type: 'page_break', label: 'فاصل صفحات', icon: 'fa-columns', category: 'layout' },
      { type: 'cta_button', label: 'زر إجراء (CTA)', icon: 'fa-hand-pointer', category: 'layout' },
      { type: 'alert_box', label: 'صندوق تنبيه', icon: 'fa-bell', category: 'layout' },
      { type: 'accordion', label: 'أسئلة شائعة (FAQ)', icon: 'fa-chevron-down', category: 'layout' },
      { type: 'countdown', label: 'عد تنازلي', icon: 'fa-hourglass-half', category: 'layout' },
      { type: 'progress_bar', label: 'شريط تقدم', icon: 'fa-tasks', category: 'layout' },
  ];

  const addField = (type: FieldType) => {
    let defaultLabel = 'عنصر جديد';
    let defaultOptions: string[] | undefined = undefined;
    let defaultPlaceholder = undefined;
    let defaultValue = undefined;

    // Smart Defaults
    if (type === 'cta_button') { defaultLabel = 'اضغط هنا'; defaultValue = '#'; }
    if (type === 'video') { defaultLabel = 'فيديو توضيحي'; defaultValue = 'https://www.youtube.com/embed/dQw4w9WgXcQ'; }
    if (type === 'map') { defaultLabel = 'موقع العيادة'; defaultValue = 'https://www.google.com/maps/embed?...'; }
    if (type === 'page_break') { defaultLabel = '--- صفحة جديدة ---'; }
    if (type === 'alert_box') { defaultLabel = 'تنبيه هام'; defaultValue = 'info'; defaultPlaceholder = 'محتوى التنبيه...'; }
    if (type === 'accordion') { defaultLabel = 'قائمة الأسئلة'; defaultOptions = ['سؤال 1|إجابة السؤال الأول', 'سؤال 2|إجابة السؤال الثاني']; }
    if (type === 'countdown') { defaultLabel = 'انتهاء العرض'; defaultValue = new Date(Date.now() + 86400000).toISOString().split('T')[0]; }
    if (type === 'html_block') { defaultLabel = 'كود مخصص'; defaultValue = '<div class="p-4 bg-gray-100">محتوى مخصص</div>'; }
    if (type === 'progress_bar') { defaultLabel = 'نسبة الاكتمال'; defaultValue = '50'; }
    if (['select', 'radio', 'checkbox', 'multiselect', 'social_share'].includes(type)) { 
        defaultOptions = ['خيار 1', 'خيار 2', 'خيار 3']; 
        defaultLabel = 'اختر من القائمة'; 
    }
    if (type === 'header') defaultLabel = 'عنوان رئيسي';
    
    const newField: FormField = {
      id: `f_${Date.now()}_${Math.floor(Math.random() * 1000)}`,
      type,
      label: defaultLabel,
      required: false,
      width: 'full',
      options: defaultOptions,
      placeholder: defaultPlaceholder,
      defaultValue: defaultValue,
      animation: 'none'
    };
    setFields([...fields, newField]);
    setSelectedFieldId(newField.id);
  };

  const updateField = (id: string, key: keyof FormField, value: any) => {
      setFields(fields.map(f => f.id === id ? { ...f, [key]: value } : f));
  };

  const removeField = (id: string) => {
      setFields(fields.filter(f => f.id !== id));
      if (selectedFieldId === id) setSelectedFieldId(null);
  };

  const moveField = (index: number, direction: 'up' | 'down') => {
      if (direction === 'up' && index === 0) return;
      if (direction === 'down' && index === fields.length - 1) return;
      
      const newFields = [...fields];
      const temp = newFields[index];
      newFields[index] = newFields[index + (direction === 'up' ? -1 : 1)];
      newFields[index + (direction === 'up' ? -1 : 1)] = temp;
      setFields(newFields);
  };

  const handleSaveForm = () => {
    if (!formTitle || fields.length === 0) {
        addToast('يرجى كتابة عنوان النموذج وإضافة حقول', 'error');
        return;
    }

    const newForm: CustomForm = {
        id: `form_${Date.now()}`,
        title: formTitle,
        description: formDesc,
        targetLocation,
        clinicType: isDoctorMode ? clinicType : targetClinic,
        fields: fields,
        isSystem: false,
        createdAt: new Date().toISOString().split('T')[0]
    };

    addCustomForm(newForm);
    addToast('تم حفظ النموذج وتعيين موقعه بنجاح', 'success');
    closeModal();
  };

  const closeModal = () => {
      setIsModalOpen(false);
      setFormTitle('');
      setFormDesc('');
      setFields([]);
      setTargetLocation('menu');
      setSelectedFieldId(null);
  };

  // --- Preview Logic ---
  const handlePreviewOpen = () => {
      setPreviewStep(0);
      setIsPreviewOpen(true);
  };

  const previewSteps = fields.reduce((acc, field) => {
      if (field.type === 'page_break') {
          acc.push([]);
      } else {
          acc[acc.length - 1].push(field);
      }
      return acc;
  }, [[]] as FormField[][]);

  const selectedField = fields.find(f => f.id === selectedFieldId);

  const categories = {
      data: 'جمع البيانات',
      inputs: 'أدوات الاختيار',
      media: 'الوسائط والميديا',
      layout: 'تخطيط وتنسيق'
  };

  // Filter forms if doctor
  const displayedForms = isDoctorMode 
    ? customForms.filter(f => f.clinicType === clinicType || f.clinicType === 'all')
    : customForms;

  return (
    <div className="space-y-6">
      <div className="flex justify-between items-center mb-6">
        <div>
          <h2 className="text-2xl font-bold text-gray-900">{isDoctorMode ? 'تخصيص النماذج' : 'باني النماذج (Form Builder)'}</h2>
          <p className="text-gray-600">{isDoctorMode ? 'إنشاء نماذج خاصة لعيادتك' : 'تصميم صفحات تفاعلية، استبيانات، ونماذج مخصصة'}</p>
        </div>
        <button 
          onClick={() => setIsModalOpen(true)}
          className="bg-primary hover:bg-primary-dark text-white px-5 py-2.5 rounded-xl shadow-md transition-colors flex items-center gap-2"
        >
          <i className="fas fa-plus-circle"></i>
          <span>إنشاء نموذج جديد</span>
        </button>
      </div>

      {/* Forms Grid */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
        {displayedForms.map((form) => (
            <div key={form.id} className="bg-white rounded-2xl shadow-sm border border-gray-200 p-6 group hover:shadow-lg transition-all relative overflow-hidden">
                <div className={`absolute top-0 right-0 w-2 h-full ${form.isSystem ? 'bg-gray-400' : 'bg-gradient-to-b from-primary to-blue-600'}`}></div>
                <div className="flex justify-between items-start mb-3 pl-2">
                    <div>
                        <h3 className="font-bold text-lg text-gray-900">{form.title}</h3>
                        <p className="text-xs text-gray-600 line-clamp-1 mt-1">{form.description || 'لا يوجد وصف'}</p>
                    </div>
                    {!form.isSystem && (
                        <button 
                            onClick={() => {
                                if(window.confirm('حذف النموذج؟')) deleteCustomForm(form.id);
                            }} 
                            className="text-gray-400 hover:text-red-600 transition-colors"
                        >
                            <i className="fas fa-trash"></i>
                        </button>
                    )}
                </div>
                <div className="flex flex-wrap gap-2 mb-4">
                    <span className="text-xs bg-blue-50 text-blue-700 px-2 py-1 rounded border border-blue-100 font-medium">
                        {form.clinicType === 'all' ? 'جميع العيادات' : form.clinicType}
                    </span>
                    <span className="text-xs bg-purple-50 text-purple-700 px-2 py-1 rounded border border-purple-100 font-medium">
                        الموقع: {form.targetLocation === 'menu' ? 'القائمة الرئيسية' : form.targetLocation === 'sidebar' ? 'القائمة الجانبية' : 'صفحة مستقلة'}
                    </span>
                </div>
                <div className="flex justify-between items-center text-xs text-gray-500 border-t border-gray-100 pt-3">
                    <span>{form.fields.length} حقول</span>
                    <span>{form.createdAt}</span>
                </div>
            </div>
        ))}
        {displayedForms.length === 0 && (
            <div className="col-span-full py-12 text-center text-gray-400 border-2 border-dashed border-gray-200 rounded-xl">
                لا توجد نماذج مخصصة
            </div>
        )}
      </div>

      {/* Builder Modal */}
      {isModalOpen && (
        <div className="fixed inset-0 z-50 flex items-center justify-center bg-gray-900/60 backdrop-blur-sm p-4">
            <div className="bg-gray-50 w-full max-w-7xl h-[95vh] rounded-2xl shadow-2xl flex flex-col overflow-hidden animate-slide-up">
                
                {/* Header */}
                <div className="bg-white border-b border-gray-200 p-4 flex justify-between items-center shrink-0">
                    <div className="flex items-center gap-4 flex-1">
                        <div className="w-10 h-10 bg-primary/10 rounded-lg flex items-center justify-center text-primary text-xl">
                            <i className="fas fa-layer-group"></i>
                        </div>
                        <div className="flex-1 space-y-1">
                            <input 
                                type="text" 
                                placeholder="عنوان النموذج/الصفحة" 
                                className="text-lg font-bold text-gray-900 bg-transparent border-b border-transparent focus:border-primary focus:ring-0 w-full p-0 placeholder-gray-400"
                                value={formTitle}
                                onChange={e => setFormTitle(e.target.value)}
                            />
                            <div className="flex gap-4">
                                <input 
                                    type="text" 
                                    placeholder="وصف قصير..." 
                                    className="text-xs text-gray-600 bg-transparent border-none focus:ring-0 w-1/2 p-0 placeholder-gray-400"
                                    value={formDesc}
                                    onChange={e => setFormDesc(e.target.value)}
                                />
                                <div className="flex items-center gap-2 text-xs">
                                    <label className="text-gray-500 font-bold">مكان الظهور:</label>
                                    <select 
                                        value={targetLocation} 
                                        onChange={e => setTargetLocation(e.target.value)}
                                        className="bg-gray-100 border-none rounded py-0 px-2 text-gray-700 text-xs focus:ring-0 cursor-pointer hover:bg-gray-200"
                                    >
                                        <option value="menu">القائمة الرئيسية (Menu)</option>
                                        <option value="sidebar">القائمة الجانبية (Sidebar)</option>
                                        <option value="standalone">صفحة مستقلة (Link Only)</option>
                                    </select>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div className="flex items-center gap-3">
                        <button 
                            onClick={handlePreviewOpen}
                            className="bg-indigo-50 text-indigo-700 px-4 py-2 rounded-lg font-bold hover:bg-indigo-100 transition-colors flex items-center gap-2 border border-indigo-100"
                        >
                            <i className="fas fa-play text-xs"></i> معاينة
                        </button>
                        <button onClick={handleSaveForm} className="bg-primary text-white px-6 py-2 rounded-lg font-bold hover:bg-primary-dark shadow-md flex items-center gap-2">
                            <i className="fas fa-save"></i> حفظ
                        </button>
                        <button onClick={closeModal} className="text-gray-400 hover:text-red-500 px-3 transition-colors">
                            <i className="fas fa-times text-2xl"></i>
                        </button>
                    </div>
                </div>

                {/* Body */}
                <div className="flex flex-1 overflow-hidden">
                    
                    {/* Left Sidebar: Toolbox */}
                    <div className="w-72 bg-white border-l border-gray-200 p-4 overflow-y-auto shrink-0 custom-scrollbar">
                        {Object.entries(categories).map(([catKey, catLabel]) => (
                            <div key={catKey} className="mb-6">
                                <h4 className="font-bold text-gray-800 mb-3 text-xs uppercase tracking-wider bg-gray-100 p-2 rounded border border-gray-200">{catLabel}</h4>
                                <div className="grid grid-cols-2 gap-2">
                                    {fieldTypes.filter(ft => ft.category === catKey).map(ft => (
                                        <button 
                                            key={ft.type}
                                            onClick={() => addField(ft.type)}
                                            className="flex flex-col items-center justify-center p-3 rounded-lg border border-gray-200 hover:border-primary hover:bg-blue-50/50 transition-all group h-20 bg-white shadow-sm"
                                        >
                                            <i className={`fas ${ft.icon} text-lg text-gray-400 group-hover:text-primary mb-2`}></i>
                                            <span className="text-[10px] font-bold text-gray-600 group-hover:text-primary text-center leading-tight">{ft.label}</span>
                                        </button>
                                    ))}
                                </div>
                            </div>
                        ))}
                    </div>

                    {/* Center: Canvas */}
                    <div className="flex-1 bg-gray-100 p-8 overflow-y-auto custom-scrollbar">
                        <div className="max-w-4xl mx-auto bg-white min-h-[600px] shadow-sm border border-gray-200 rounded-xl p-8 relative">
                            {fields.length === 0 && (
                                <div className="absolute inset-0 flex flex-col items-center justify-center text-gray-400 pointer-events-none">
                                    <div className="w-24 h-24 bg-gray-50 rounded-full flex items-center justify-center mb-4 border border-gray-100">
                                        <i className="fas fa-magic text-4xl opacity-20 text-primary"></i>
                                    </div>
                                    <p className="text-xl font-bold text-gray-600">مساحة العمل فارغة</p>
                                    <p className="text-sm text-gray-500 mt-2">اختر العناصر من القائمة اليمنى لبناء النموذج</p>
                                </div>
                            )}
                            
                            <div className="space-y-4">
                                {fields.map((field, idx) => (
                                    <div 
                                        key={field.id}
                                        onClick={() => setSelectedFieldId(field.id)}
                                        className={`relative group p-4 rounded-xl border-2 transition-all cursor-pointer ${
                                            selectedFieldId === field.id ? 'border-primary bg-blue-50/20' : 'border-transparent hover:border-gray-300 hover:bg-gray-50'
                                        }`}
                                    >
                                        {/* Simple Builder Representation */}
                                        <div className="pointer-events-none select-none">
                                            {field.type === 'header' && <h3 className="text-2xl font-bold text-gray-900 border-b pb-2">{field.label}</h3>}
                                            {field.type === 'page_break' && <div className="flex items-center gap-4 text-gray-400"><div className="h-px bg-gray-300 flex-1"></div><span className="text-xs font-bold uppercase bg-gray-200 px-3 py-1 rounded text-gray-600 border border-gray-300">فاصل صفحات</span><div className="h-px bg-gray-300 flex-1"></div></div>}
                                            
                                            {/* ... (Render other types same as before) ... */}
                                            {!['header', 'page_break', 'video', 'map', 'countdown', 'accordion', 'html_block'].includes(field.type) && (
                                                <>
                                                    <label className="block text-sm font-bold text-gray-800 mb-1">{field.label} {field.required && <span className="text-red-600">*</span>}</label>
                                                    <div className="h-10 bg-white border border-gray-300 rounded w-full flex items-center px-3 text-gray-500 text-sm">
                                                        {field.placeholder || `محتوى حقل ${field.type}`}
                                                    </div>
                                                </>
                                            )}
                                        </div>

                                        <div className="absolute left-2 top-2 hidden group-hover:flex gap-1 bg-white shadow-md rounded-lg p-1 z-10 border border-gray-100">
                                            <button onClick={(e) => { e.stopPropagation(); moveField(idx, 'up'); }} className="p-1.5 hover:text-blue-600 transition-colors"><i className="fas fa-arrow-up"></i></button>
                                            <button onClick={(e) => { e.stopPropagation(); moveField(idx, 'down'); }} className="p-1.5 hover:text-blue-600 transition-colors"><i className="fas fa-arrow-down"></i></button>
                                            <button onClick={(e) => { e.stopPropagation(); removeField(field.id); }} className="p-1.5 hover:text-red-600 transition-colors"><i className="fas fa-trash"></i></button>
                                        </div>
                                    </div>
                                ))}
                            </div>
                        </div>
                    </div>

                    {/* Right Sidebar */}
                    <div className="w-80 bg-white border-r border-gray-200 p-4 overflow-y-auto shrink-0 custom-scrollbar">
                        <h4 className="font-bold text-gray-800 mb-4 text-sm uppercase tracking-wider border-b border-gray-100 pb-2">خصائص العنصر</h4>
                        
                        {selectedField ? (
                            <div className="space-y-5 animate-fade-in">
                                <div>
                                    <label className="block text-xs font-bold text-gray-700 mb-1.5">العنوان / النص الظاهر</label>
                                    <input 
                                        type="text"
                                        value={selectedField.label}
                                        onChange={e => updateField(selectedField.id, 'label', e.target.value)}
                                        className="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:border-primary focus:ring-1 focus:ring-primary text-gray-900 bg-white"
                                    />
                                </div>
                                <div>
                                    <label className="block text-xs font-bold text-gray-700 mb-1.5">نص توضيحي (Placeholder)</label>
                                    <input 
                                        type="text"
                                        value={selectedField.placeholder || ''}
                                        onChange={e => updateField(selectedField.id, 'placeholder', e.target.value)}
                                        className="w-full px-3 py-2 border border-gray-300 rounded-lg text-sm focus:border-primary focus:ring-1 focus:ring-primary text-gray-900 bg-white"
                                    />
                                </div>
                                <div className="flex items-center gap-2 pt-4 border-t border-gray-100">
                                    <input 
                                        type="checkbox" 
                                        id="req-check"
                                        checked={selectedField.required} 
                                        onChange={e => updateField(selectedField.id, 'required', e.target.checked)}
                                        className="rounded text-primary focus:ring-primary w-4 h-4 bg-white"
                                    />
                                    <label htmlFor="req-check" className="text-sm font-bold text-gray-700 cursor-pointer">هذا الحقل إجباري (Required)</label>
                                </div>
                            </div>
                        ) : (
                            <div className="text-center text-gray-400 py-10">
                                <i className="fas fa-mouse-pointer mb-3 text-3xl opacity-50"></i>
                                <p className="text-sm font-medium text-gray-500">اضغط على أي عنصر في ساحة العمل لتعديل خصائصه</p>
                            </div>
                        )}
                    </div>
                </div>
            </div>
        </div>
      )}

      {/* Live Preview Modal */}
      {isPreviewOpen && (
          <Modal isOpen={isPreviewOpen} onClose={() => setIsPreviewOpen(false)} title="معاينة حية للنموذج">
              <div className="bg-gray-100 p-6 rounded-xl max-h-[85vh] overflow-y-auto">
                  <div className="bg-white rounded-3xl shadow-xl border border-gray-200 overflow-hidden max-w-4xl mx-auto">
                        <div className="bg-white border-b border-gray-200 p-6 flex justify-between items-center sticky top-0 z-10">
                            <h2 className="text-2xl font-extrabold text-gray-900">{formTitle || 'عنوان النموذج'}</h2>
                            {previewSteps.length > 1 && (
                                <div className="flex items-center gap-3">
                                    <span className="text-sm font-bold text-primary">خطوة {previewStep + 1} من {previewSteps.length}</span>
                                    <div className="w-32 h-2.5 bg-gray-100 rounded-full overflow-hidden border border-gray-200">
                                        <div className="h-full bg-primary transition-all duration-500" style={{ width: `${((previewStep + 1) / previewSteps.length) * 100}%` }}></div>
                                    </div>
                                </div>
                            )}
                        </div>
                        <div className="p-8">
                            <div className="grid grid-cols-1 md:grid-cols-12 gap-6">
                                {previewSteps[previewStep]?.map(field => {
                                    const colSpan = field.width === 'full' ? 'md:col-span-12' : field.width === 'half' ? 'md:col-span-6' : 'md:col-span-4';
                                    
                                    if (field.type === 'header') return <h3 key={field.id} className={`text-2xl font-bold text-gray-800 border-b-2 border-gray-100 pb-3 mb-2 ${colSpan}`}>{field.label}</h3>;
                                    if (field.type === 'paragraph') return <p key={field.id} className={`text-gray-700 leading-relaxed ${colSpan}`}>{field.label}</p>;
                                    // ... other preview renders ...
                                    return (
                                        <div key={field.id} className={`${colSpan}`}>
                                            <label className="block text-sm font-bold text-gray-800 mb-2">{field.label} {field.required && <span className="text-red-600">*</span>}</label>
                                            <input type="text" className="w-full px-4 py-3 border border-gray-300 rounded-xl bg-white text-gray-900" placeholder={field.placeholder} />
                                        </div>
                                    );
                                })}
                            </div>
                            <div className="mt-10 flex justify-between pt-6 border-t border-gray-100">
                                <button disabled={previewStep === 0} onClick={() => setPreviewStep(p => p - 1)} className={`px-6 py-2.5 rounded-xl font-bold transition-colors ${previewStep === 0 ? 'text-gray-300 cursor-not-allowed' : 'text-gray-700 bg-gray-100 hover:bg-gray-200'}`}>سابق</button>
                                {previewStep < previewSteps.length - 1 ? (
                                    <button onClick={() => setPreviewStep(p => p + 1)} className="bg-primary text-white px-8 py-2.5 rounded-xl font-bold shadow-lg hover:bg-primary-dark">التالي</button>
                                ) : (
                                    <button className="bg-green-600 text-white px-8 py-2.5 rounded-xl font-bold shadow-lg hover:bg-green-700">إرسال (تجربة)</button>
                                )}
                            </div>
                        </div>
                  </div>
              </div>
          </Modal>
      )}
    </div>
  );
};
