import React, { createContext, useContext, useState, useEffect } from 'react';
import { 
  mockTenants, mockPatients, mockAppointments, mockInvoices, 
  mockPlans, mockServices, mockTemplates,
  mockLabOrders, mockBirthRecords, mockModules, mockCustomForms, mockDynamicPages,
  mockInventory, mockPrescriptions, mockMedicalReports, mockAuditLogs, mockSupportTickets
} from '../mockData';
import { 
  Tenant, Patient, Appointment, Invoice, Plan, Visit, Service, Template, ClinicType, 
  LabOrder, BirthRecord, AppSettings, Notification, ModuleFeature, CustomForm, DynamicPage,
  InventoryItem, Prescription, ClinicStaff, MedicalReport, SiteContent, AuditLog, SupportTicket, FieldConfig,
  SyncAction
} from '../types';
import { api } from '../services/api';

interface DataContextType {
  clinicType: ClinicType;
  setClinicType: (type: ClinicType) => void;
  appSettings: AppSettings;
  updateAppSettings: (settings: AppSettings) => void;
  tenants: Tenant[];
  addTenant: (tenant: Tenant) => void;
  updateTenant: (tenant: Tenant) => void;
  deleteTenant: (id: string) => void;
  activateTenant: (email: string, code: string) => boolean;
  updateTenantSiteContent: (content: SiteContent) => void; 
  patients: Patient[];
  addPatient: (patient: Patient) => void;
  updatePatient: (patient: Patient) => void; 
  deletePatient: (id: string) => void;
  appointments: Appointment[];
  addAppointment: (appointment: Appointment) => void;
  updateAppointmentStatus: (id: string, status: 'confirmed' | 'cancelled' | 'pending' | 'completed') => void;
  invoices: Invoice[];
  addInvoice: (invoice: Invoice) => void;
  plans: Plan[];
  updatePlan: (plan: Plan) => void;
  services: Service[];
  addService: (service: Service) => void;
  updateService: (service: Service) => void;
  deleteService: (id: string) => void;
  templates: Template[];
  addTemplate: (template: Template) => void;
  updateTemplate: (template: Template) => void;
  toggleTemplateStatus: (id: string) => void;
  visits: Visit[];
  addVisit: (visit: Visit) => void;
  labOrders: LabOrder[];
  addLabOrder: (order: LabOrder) => void;
  updateLabOrder: (order: LabOrder) => void;
  birthRecords: BirthRecord[];
  addBirthRecord: (record: BirthRecord) => void;
  inventory: InventoryItem[];
  addInventoryItem: (item: InventoryItem) => void;
  updateInventoryItem: (item: InventoryItem) => void;
  deleteInventoryItem: (id: string) => void;
  prescriptions: Prescription[];
  addPrescription: (prescription: Prescription) => void;
  staff: ClinicStaff[];
  addStaff: (staff: ClinicStaff) => void;
  deleteStaff: (id: string) => void;
  medicalReports: MedicalReport[];
  addMedicalReport: (report: MedicalReport) => void;
  notifications: Notification[];
  addNotification: (title: string, message: string, type: 'info' | 'warning' | 'success' | 'error') => void;
  markNotificationAsRead: (id: number) => void;
  clearNotifications: () => void;
  modules: ModuleFeature[];
  updateModule: (module: ModuleFeature) => void;
  customForms: CustomForm[];
  addCustomForm: (form: CustomForm) => void;
  deleteCustomForm: (id: string) => void;
  dynamicPages: DynamicPage[];
  addDynamicPage: (page: DynamicPage) => void;
  updateDynamicPageOrder: (pages: DynamicPage[]) => void;
  togglePageVisibility: (id: string) => void;
  deleteDynamicPage: (id: string) => void;
  fieldConfigs: FieldConfig[];
  updateFieldConfig: (config: FieldConfig) => void;
  addFieldConfig: (config: FieldConfig) => void;
  deleteFieldConfig: (id: string) => void;
  auditLogs: AuditLog[];
  addAuditLog: (log: AuditLog) => void;
  supportTickets: SupportTicket[];
  updateTicketStatus: (id: string, status: SupportTicket['status']) => void;
  isSyncing: boolean;
  pendingSyncCount: number;
}

const DataContext = createContext<DataContextType | undefined>(undefined);

const loadState = <T,>(key: string, fallback: T): T => {
  try {
    const saved = localStorage.getItem(key);
    return saved ? JSON.parse(saved) : fallback;
  } catch (e) {
    return fallback;
  }
};

const defaultSettings: AppSettings = {
  platformName: "خالد قصرين Pro",
  supportEmail: "support@khaledq.com",
  backupEnabled: true,
  maintenanceMode: false
};

const usePersistedState = <T,>(key: string, initialValue: T): [T, React.Dispatch<React.SetStateAction<T>>] => {
    const [state, setState] = useState<T>(() => loadState(key, initialValue));
    useEffect(() => {
        localStorage.setItem(key, JSON.stringify(state));
    }, [key, state]);
    return [state, setState];
};

export const DataProvider: React.FC<{ children: React.ReactNode }> = ({ children }) => {
  const [clinicType, setClinicType] = usePersistedState<ClinicType>('clinicType', 'general');
  const [appSettings, setAppSettings] = usePersistedState<AppSettings>('appSettings', defaultSettings);
  const [tenants, setTenants] = usePersistedState<Tenant[]>('tenants', mockTenants);
  const [patients, setPatients] = usePersistedState<Patient[]>('patients', mockPatients);
  const [appointments, setAppointments] = usePersistedState<Appointment[]>('appointments', mockAppointments);
  const [invoices, setInvoices] = usePersistedState<Invoice[]>('invoices', mockInvoices);
  const [plans, setPlans] = usePersistedState<Plan[]>('plans', mockPlans);
  const [services, setServices] = usePersistedState<Service[]>('services', mockServices);
  const [templates, setTemplates] = usePersistedState<Template[]>('templates', mockTemplates);
  const [visits, setVisits] = usePersistedState<Visit[]>('visits', []);
  const [labOrders, setLabOrders] = usePersistedState<LabOrder[]>('labOrders', mockLabOrders);
  const [birthRecords, setBirthRecords] = usePersistedState<BirthRecord[]>('birthRecords', mockBirthRecords);
  const [inventory, setInventory] = usePersistedState<InventoryItem[]>('inventory', mockInventory);
  const [prescriptions, setPrescriptions] = usePersistedState<Prescription[]>('prescriptions', mockPrescriptions);
  const [staff, setStaff] = usePersistedState<ClinicStaff[]>('staff', [
      { id: 's1', name: 'د. أحمد خالد', role: 'doctor', email: 'ahmed@clinic.com', phone: '0912345678', status: 'active', permissions: ['all'] },
  ]);
  const [medicalReports, setMedicalReports] = usePersistedState<MedicalReport[]>('medicalReports', mockMedicalReports);
  const [modules, setModules] = usePersistedState<ModuleFeature[]>('modules', mockModules);
  const [customForms, setCustomForms] = usePersistedState<CustomForm[]>('customForms', mockCustomForms);
  const [dynamicPages, setDynamicPages] = usePersistedState<DynamicPage[]>('dynamicPages', mockDynamicPages);
  const [fieldConfigs, setFieldConfigs] = usePersistedState<FieldConfig[]>('fieldConfigs', []);
  const [auditLogs, setAuditLogs] = usePersistedState<AuditLog[]>('auditLogs', mockAuditLogs);
  const [supportTickets, setSupportTickets] = usePersistedState<SupportTicket[]>('supportTickets', mockSupportTickets);
  const [notifications, setNotifications] = useState<Notification[]>([]);
  const [syncQueue, setSyncQueue] = usePersistedState<SyncAction[]>('sync_queue', []);
  const [isSyncing, setIsSyncing] = useState(false);
  const [isOnline, setIsOnline] = useState(navigator.onLine);

  useEffect(() => {
    const handleOnline = () => { setIsOnline(true); processSyncQueue(); };
    const handleOffline = () => setIsOnline(false);
    window.addEventListener('online', handleOnline);
    window.addEventListener('offline', handleOffline);
    return () => {
      window.removeEventListener('online', handleOnline);
      window.removeEventListener('offline', handleOffline);
    };
  }, []);

  const processSyncQueue = async () => {
    if (syncQueue.length === 0) return;
    setIsSyncing(true);
    try {
      await api.syncBatch(syncQueue);
      setSyncQueue([]);
      addNotification('تمت المزامنة', 'تم رفع البيانات بنجاح للسحابة', 'success');
    } catch (error) {
      console.error("Sync error", error);
    } finally {
      setIsSyncing(false);
    }
  };

  const mutateData = async (collection: string, type: 'CREATE' | 'UPDATE' | 'DELETE', payload: any, localUpdater: () => void) => {
    localUpdater();
    const action: SyncAction = { id: payload.id || Date.now().toString(), type, collection, payload, timestamp: Date.now() };
    if (navigator.onLine) {
      try {
        if (type === 'CREATE') await api.create(collection, payload);
        else if (type === 'UPDATE') await api.update(collection, payload);
        else await api.delete(collection, payload.id);
      } catch (e) {
        setSyncQueue(prev => [...prev, action]);
      }
    } else {
      setSyncQueue(prev => [...prev, action]);
    }
  };

  const addNotification = (title: string, message: string, type: 'info' | 'warning' | 'success' | 'error' = 'info') => {
    setNotifications(prev => [{ id: Date.now(), title, message, type, timestamp: 'الآن', read: false }, ...prev]);
  };
  const markNotificationAsRead = (id: number) => setNotifications(prev => prev.map(n => n.id === id ? { ...n, read: true } : n));
  const clearNotifications = () => setNotifications([]);
  const updateAppSettings = (settings: AppSettings) => setAppSettings(settings);

  const addTenant = (tenant: Tenant) => mutateData('tenants', 'CREATE', tenant, () => setTenants(p => [tenant, ...p]));
  const updateTenant = (tenant: Tenant) => mutateData('tenants', 'UPDATE', tenant, () => setTenants(p => p.map(t => t.id === tenant.id ? tenant : t)));
  const deleteTenant = (id: string) => mutateData('tenants', 'DELETE', {id}, () => setTenants(p => p.filter(t => t.id !== id)));
  const activateTenant = (email: string, code: string) => {
    const idx = tenants.findIndex(t => t.email === email && t.activationCode === code);
    if (idx !== -1) {
      updateTenant({ ...tenants[idx], isActivated: true });
      return true;
    }
    return false;
  };
  const updateTenantSiteContent = (content: SiteContent) => {
    const t = tenants.find(t => t.clinicType === clinicType);
    if (t) updateTenant({ ...t, siteContent: content });
  };

  const addPatient = (p: Patient) => mutateData('patients', 'CREATE', p, () => setPatients(prev => [p, ...prev]));
  const updatePatient = (p: Patient) => mutateData('patients', 'UPDATE', p, () => setPatients(prev => prev.map(i => i.id === p.id ? p : i)));
  const deletePatient = (id: string) => mutateData('patients', 'DELETE', {id}, () => setPatients(p => p.filter(i => i.id !== id)));

  const addAppointment = (a: Appointment) => {
    mutateData('appointments', 'CREATE', a, () => setAppointments(prev => [a, ...prev]));
    addNotification('موعد جديد', `تم حجز موعد لـ ${a.patientName}`, 'info');
  };
  const updateAppointmentStatus = (id: string, status: any) => {
    const app = appointments.find(a => a.id === id);
    if (app) updatePatient({ ...app, status } as any);
  };

  const addInvoice = (inv: Invoice) => mutateData('invoices', 'CREATE', inv, () => setInvoices(p => [inv, ...p]));
  const updatePlan = (p: Plan) => mutateData('plans', 'UPDATE', p, () => setPlans(prev => prev.map(i => i.id === p.id ? p : i)));
  const addService = (s: Service) => mutateData('services', 'CREATE', s, () => setServices(p => [...p, s]));
  const updateService = (s: Service) => mutateData('services', 'UPDATE', s, () => setServices(p => p.map(i => i.id === s.id ? s : i)));
  const deleteService = (id: string) => mutateData('services', 'DELETE', {id}, () => setServices(p => p.filter(s => s.id !== id)));
  const addTemplate = (t: Template) => mutateData('templates', 'CREATE', t, () => setTemplates(p => [t, ...p]));
  const updateTemplate = (t: Template) => mutateData('templates', 'UPDATE', t, () => setTemplates(p => p.map(i => i.id === t.id ? t : i)));
  const toggleTemplateStatus = (id: string) => {
    const t = templates.find(x => x.id === id);
    if (t) updateTemplate({ ...t, isActive: !t.isActive });
  };

  const addVisit = (v: Visit) => mutateData('visits', 'CREATE', v, () => setVisits(p => [v, ...p]));
  const addLabOrder = (o: LabOrder) => mutateData('labOrders', 'CREATE', o, () => setLabOrders(p => [o, ...p]));
  const updateLabOrder = (o: LabOrder) => mutateData('labOrders', 'UPDATE', o, () => setLabOrders(p => p.map(x => x.id === o.id ? o : x)));
  const addBirthRecord = (r: BirthRecord) => mutateData('birthRecords', 'CREATE', r, () => setBirthRecords(p => [r, ...p]));
  const addInventoryItem = (i: InventoryItem) => mutateData('inventory', 'CREATE', i, () => setInventory(p => [i, ...p]));
  const updateInventoryItem = (i: InventoryItem) => mutateData('inventory', 'UPDATE', i, () => setInventory(p => p.map(x => x.id === i.id ? i : x)));
  const deleteInventoryItem = (id: string) => mutateData('inventory', 'DELETE', {id}, () => setInventory(p => p.filter(x => x.id !== id)));
  const addPrescription = (p: Prescription) => mutateData('prescriptions', 'CREATE', p, () => setPrescriptions(prev => [p, ...prev]));
  const addStaff = (s: ClinicStaff) => mutateData('staff', 'CREATE', s, () => setStaff(p => [...p, s]));
  const deleteStaff = (id: string) => mutateData('staff', 'DELETE', {id}, () => setStaff(p => p.filter(s => s.id !== id)));
  const addMedicalReport = (r: MedicalReport) => mutateData('medicalReports', 'CREATE', r, () => setMedicalReports(p => [r, ...p]));
  const updateModule = (m: ModuleFeature) => setModules(prev => prev.map(x => x.id === m.id ? m : x));
  const addCustomForm = (f: CustomForm) => setCustomForms(prev => [...prev, f]);
  const deleteCustomForm = (id: string) => setCustomForms(prev => prev.filter(f => f.id !== id));
  const addDynamicPage = (p: DynamicPage) => setDynamicPages(prev => [...prev, p]);
  const updateDynamicPageOrder = (pages: DynamicPage[]) => setDynamicPages(pages);
  const togglePageVisibility = (id: string) => setDynamicPages(prev => prev.map(p => p.id === id ? { ...p, isVisible: !p.isVisible } : p));
  const deleteDynamicPage = (id: string) => setDynamicPages(prev => prev.filter(p => p.id !== id));
  const updateFieldConfig = (c: FieldConfig) => setFieldConfigs(prev => prev.map(f => f.id === c.id ? c : f));
  const addFieldConfig = (c: FieldConfig) => setFieldConfigs(prev => [...prev, c]);
  const deleteFieldConfig = (id: string) => setFieldConfigs(prev => prev.filter(f => f.id !== id));
  const addAuditLog = (l: AuditLog) => mutateData('auditLogs', 'CREATE', l, () => setAuditLogs(p => [l, ...p]));
  const updateTicketStatus = (id: string, status: any) => {
    const t = supportTickets.find(s => s.id === id);
    if (t) mutateData('supportTickets', 'UPDATE', { ...t, status }, () => setSupportTickets(prev => prev.map(x => x.id === id ? { ...t, status } : x)));
  };

  return (
    <DataContext.Provider value={{
      clinicType, setClinicType, appSettings, updateAppSettings,
      tenants, addTenant, updateTenant, deleteTenant, activateTenant, updateTenantSiteContent,
      patients, addPatient, updatePatient, deletePatient,
      appointments, addAppointment, updateAppointmentStatus,
      invoices, addInvoice, plans, updatePlan, services, addService, updateService, deleteService,
      templates, addTemplate, updateTemplate, toggleTemplateStatus,
      visits, addVisit, labOrders, addLabOrder, updateLabOrder, birthRecords, addBirthRecord,
      inventory, addInventoryItem, updateInventoryItem, deleteInventoryItem,
      prescriptions, addPrescription, staff, addStaff, deleteStaff, medicalReports, addMedicalReport,
      notifications, addNotification, markNotificationAsRead, clearNotifications,
      modules, updateModule, customForms, addCustomForm, deleteCustomForm,
      dynamicPages, addDynamicPage, updateDynamicPageOrder, togglePageVisibility, deleteDynamicPage,
      fieldConfigs, updateFieldConfig, addFieldConfig, deleteFieldConfig,
      auditLogs, addAuditLog, supportTickets, updateTicketStatus,
      isSyncing, pendingSyncCount: syncQueue.length
    }}>
      {children}
    </DataContext.Provider>
  );
};

export const useData = () => {
  const context = useContext(DataContext);
  if (context === undefined) throw new Error('useData must be used within a DataProvider');
  return context;
};