import React, { useState, useRef, useEffect } from 'react';
import { GoogleGenAI } from "@google/genai";

interface Message {
  id: number;
  text: string;
  isBot: boolean;
}

export const ChatWidget: React.FC = () => {
  const [isOpen, setIsOpen] = useState(false);
  const [messages, setMessages] = useState<Message[]>([
    { id: 1, text: 'مرحباً بك! أنا مساعد "خالد قصرين" الذكي. يمكنني إرشادك تقنياً في استخدام المنصة، أو تقديم تحليل أولي لأعراضك الصحية. كيف يمكنني مساعدتك اليوم؟ ✨', isBot: true }
  ]);
  const [inputText, setInputText] = useState('');
  const [isTyping, setIsTyping] = useState(false);
  const messagesEndRef = useRef<HTMLDivElement>(null);

  const scrollToBottom = () => {
    messagesEndRef.current?.scrollIntoView({ behavior: "smooth" });
  };

  useEffect(() => {
    if (isOpen) scrollToBottom();
  }, [messages, isOpen]);

  const handleSendMessage = async () => {
    if (!inputText.trim()) return;

    const userText = inputText;
    setMessages(prev => [...prev, { id: Date.now(), text: userText, isBot: false }]);
    setInputText('');
    setIsTyping(true);

    try {
      const ai = new GoogleGenAI({ apiKey: process.env.API_KEY });
      const response = await ai.models.generateContent({
        model: 'gemini-3-flash-preview',
        contents: userText,
        config: {
          systemInstruction: `أنت 'مساعد خالد قصرين الذكي'، خبير في منصة 'نظام خالد قصرين Pro' ومستشار طبي أولي.

          مهمتك محددة بدقة في مسارين فقط، وأي شيء آخر ترفضه:

          1. المسار التقني (دليل المستخدم):
          - اشرح للمستخدمين كيفية الحجز (يتم عبر الرابط العام للعيادة /site/domain).
          - اشرح للأطباء كيفية إدارة 'سجل المرضى'، 'المواعيد'، وإصدار 'الوصفات الطبية' و'التقارير'.
          - إذا سأل عن 'كيف أحجز؟' قل له: 'يمكنك التوجه لصفحة العيادة العامة، اختيار التاريخ والوقت المناسبين، ثم تأكيد الحجز برقم هاتفك'.

          2. المسار الطبي (تحليل الأعراض):
          - إذا وصف المستخدم أعراضاً (مثل: صداع في الجبهة، ألم أسنان، غثيان)، قم بتحليلها منطقياً.
          - مثال: صداع الجبهة قد يشير لجيوب أنفية أو إجهاد بصري.
          - قدم نصائح أولية: (شرب ماء، راحة في غرفة مظلمة، وضع كمادات).
          - إلزامي: ابدأ الرد الطبي بـ: '⚠️ تنبيه: هذا تحليل مدعوم بالذكاء الاصطناعي للإرشاد فقط، ولا يعتبر تشخيصاً طبياً نهائياً. يرجى زيارة الطبيب المختص.'

          قواعد صارمة:
          - ارفض الأسئلة السياسية، الرياضية، العامة، أو الدردشة غير الهادفة.
          - الرد في حال الخروج عن النص: 'نعتذر منك، أنا مبرمج حصراً للمساعدة في الأمور التقنية للنظام أو تقديم استشارات طبية أولية.'
          - لغة الرد: عربية فصحى بسيطة ومهنية.`,
          temperature: 0.1,
        }
      });

      const botText = response.text || 'عذراً، لم أتمكن من معالجة طلبك.';
      setMessages(prev => [...prev, { id: Date.now() + 1, text: botText, isBot: true }]);
    } catch (error) {
      console.error("AI Error:", error);
      setMessages(prev => [...prev, { id: Date.now() + 1, text: 'حدث خطأ في الاتصال، يرجى المحاولة لاحقاً.', isBot: true }]);
    } finally {
      setIsTyping(false);
    }
  };

  return (
    <div className="fixed bottom-6 left-6 z-[100] flex flex-col items-end no-print">
      {isOpen && (
        <div className="mb-4 w-[350px] md:w-[420px] bg-white rounded-3xl shadow-2xl border border-slate-200 overflow-hidden flex flex-col animate-slide-up h-[550px]">
          <div className="bg-gradient-to-r from-primary to-blue-700 p-4 flex justify-between items-center text-white shadow-lg">
            <div className="flex items-center gap-3">
              <div className="w-10 h-10 bg-white/20 rounded-xl flex items-center justify-center border border-white/30">
                <i className="fas fa-robot text-xl"></i>
              </div>
              <div>
                <h4 className="font-bold text-sm">مساعد خالد قصرين</h4>
                <span className="text-[10px] opacity-80 flex items-center gap-1">
                  <span className="w-1.5 h-1.5 bg-green-400 rounded-full animate-pulse"></span> متصل الآن
                </span>
              </div>
            </div>
            <button onClick={() => setIsOpen(false)} className="hover:rotate-90 transition-transform p-1">
              <i className="fas fa-times text-xl"></i>
            </button>
          </div>
          
          <div className="flex-1 p-4 overflow-y-auto bg-slate-50/50 flex flex-col gap-4 custom-scrollbar">
            {messages.map((msg) => (
              <div key={msg.id} className={`flex ${msg.isBot ? 'justify-start' : 'justify-end'}`}>
                <div className={`max-w-[90%] p-3.5 rounded-2xl text-sm shadow-sm leading-relaxed whitespace-pre-wrap ${
                  msg.isBot ? 'bg-white text-slate-700 rounded-tl-none border border-slate-100' : 'bg-primary text-white rounded-tr-none'
                }`}>
                  {msg.text}
                </div>
              </div>
            ))}
            {isTyping && (
              <div className="flex justify-start">
                <div className="bg-white p-3 rounded-2xl rounded-tl-none border border-slate-100 flex gap-1">
                  <span className="w-1.5 h-1.5 bg-primary/40 rounded-full animate-bounce"></span>
                  <span className="w-1.5 h-1.5 bg-primary/40 rounded-full animate-bounce [animation-delay:0.2s]"></span>
                  <span className="w-1.5 h-1.5 bg-primary/40 rounded-full animate-bounce [animation-delay:0.4s]"></span>
                </div>
              </div>
            )}
            <div ref={messagesEndRef} />
          </div>

          <div className="p-4 bg-white border-t border-slate-100">
            <div className="flex gap-2">
                <input 
                  type="text" 
                  placeholder="اسأل عن حجز، ميزة، أو استشارة..." 
                  className="flex-1 px-4 py-2.5 bg-slate-50 border border-slate-200 rounded-xl text-sm focus:outline-none focus:ring-2 focus:ring-primary/20 transition-all text-slate-900"
                  value={inputText}
                  onChange={(e) => setInputText(e.target.value)}
                  onKeyDown={(e) => e.key === 'Enter' && handleSendMessage()}
                />
                <button 
                  onClick={handleSendMessage}
                  disabled={!inputText.trim() || isTyping}
                  className="w-10 h-10 bg-primary text-white rounded-xl hover:bg-primary-dark transition-all flex items-center justify-center disabled:opacity-50 shadow-lg"
                >
                  <i className={`fas ${isTyping ? 'fa-spinner fa-spin' : 'fa-paper-plane'}`}></i>
                </button>
            </div>
            <div className="flex justify-center items-center gap-1 mt-2">
                <i className="fas fa-user-shield text-[8px] text-gray-400"></i>
                <p className="text-[9px] text-gray-400 italic">نظام ذكاء اصطناعي آمن ومغلق التخصص.</p>
            </div>
          </div>
        </div>
      )}

      <button 
        onClick={() => setIsOpen(!isOpen)}
        className={`w-16 h-16 rounded-full shadow-2xl flex items-center justify-center transition-all hover:scale-110 active:scale-95 group relative border-4 border-white ${isOpen ? 'bg-slate-800 text-white' : 'bg-primary text-white'}`}
      >
        <i className={`fas ${isOpen ? 'fa-times' : 'fa-hand-holding-heart'} text-2xl group-hover:rotate-12 transition-transform`}></i>
        {!isOpen && (
            <span className="absolute -top-1 -right-1 w-5 h-5 bg-red-500 rounded-full border-2 border-white animate-bounce flex items-center justify-center text-[10px] font-bold text-white shadow-lg">!</span>
        )}
      </button>
    </div>
  );
};