
import React, { useState } from 'react';
import { StatCard } from '../../components/StatCard';

export const AdminReports: React.FC = () => {
  const [period, setPeriod] = useState<'monthly' | 'yearly'>('monthly');

  // Mock data for charts - Dynamic based on period
  const data = period === 'monthly' 
    ? {
        income: [12, 19, 15, 25, 22, 30],
        labels: ['يناير', 'فبراير', 'مارس', 'أبريل', 'مايو', 'يونيو'],
        totalIncome: '$150,000',
        growth: '8.5%',
        churn: '1.2%'
      }
    : {
        income: [120, 150, 180, 220, 250, 300],
        labels: ['2019', '2020', '2021', '2022', '2023', '2024'],
        totalIncome: '$1,200,000',
        growth: '25%',
        churn: '0.8%'
      };

  const maxIncome = Math.max(...data.income);

  return (
    <div className="space-y-6">
      <div className="flex justify-between items-center">
          <h2 className="text-2xl font-bold text-gray-800">التقارير والتحليلات</h2>
          <div className="flex bg-gray-100 p-1 rounded-lg">
              <button 
                onClick={() => setPeriod('monthly')}
                className={`px-4 py-2 text-sm font-medium rounded-md transition-all ${period === 'monthly' ? 'bg-white shadow text-primary' : 'text-gray-500 hover:text-gray-700'}`}
              >
                  شهري
              </button>
              <button 
                onClick={() => setPeriod('yearly')}
                className={`px-4 py-2 text-sm font-medium rounded-md transition-all ${period === 'yearly' ? 'bg-white shadow text-primary' : 'text-gray-500 hover:text-gray-700'}`}
              >
                  سنوي
              </button>
          </div>
      </div>
      
      {/* Summary Cards */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-6 animate-fade-in">
        <StatCard title={period === 'monthly' ? "صافي الدخل الشهري" : "صافي الدخل السنوي"} value={data.totalIncome} icon="fa-chart-line" color="success" trend="+12% مقارنة بالفترة السابقة" />
        <StatCard title="متوسط نمو المشتركين" value={data.growth} icon="fa-user-plus" color="primary" trend="تصاعدي" />
        <StatCard title="معدل إلغاء الاشتراك" value={data.churn} icon="fa-user-minus" color="danger" trend="منخفض" />
      </div>

      {/* Bar Chart Simulation */}
      <div className="bg-white p-6 rounded-2xl shadow-sm border border-gray-100">
        <h3 className="text-lg font-bold text-gray-800 mb-6">نمو الإيرادات ({period === 'monthly' ? 'آخر 6 أشهر' : 'آخر 6 سنوات'})</h3>
        <div className="flex items-end justify-between h-64 gap-2">
            {data.income.map((val, idx) => (
                <div key={idx} className="flex-1 flex flex-col items-center gap-2 group cursor-pointer">
                    <div className="relative w-full flex justify-center h-full items-end">
                        <div 
                            style={{ height: `${(val / maxIncome) * 100}%` }} 
                            className="w-full max-w-[40px] bg-primary/80 group-hover:bg-primary rounded-t-lg transition-all duration-500 relative"
                        >
                            <span className="absolute -top-8 left-1/2 -translate-x-1/2 bg-gray-800 text-white text-xs py-1 px-2 rounded opacity-0 group-hover:opacity-100 transition-opacity shadow-lg z-10 whitespace-nowrap">
                                ${val}k
                            </span>
                        </div>
                    </div>
                    <span className="text-xs text-gray-500 font-medium">{data.labels[idx]}</span>
                </div>
            ))}
        </div>
      </div>

      {/* Subscription Distribution */}
      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <div className="bg-white p-6 rounded-2xl shadow-sm border border-gray-100">
            <h3 className="text-lg font-bold text-gray-800 mb-4">توزيع الباقات</h3>
            <div className="space-y-4">
                <div className="flex items-center justify-between">
                    <span className="flex items-center gap-2"><div className="w-3 h-3 rounded-full bg-blue-500"></div> باقة العيادات</span>
                    <span className="font-bold">45%</span>
                </div>
                <div className="w-full bg-gray-100 rounded-full h-2">
                    <div className="bg-blue-500 h-2 rounded-full" style={{ width: '45%' }}></div>
                </div>

                <div className="flex items-center justify-between">
                    <span className="flex items-center gap-2"><div className="w-3 h-3 rounded-full bg-green-500"></div> الباقة المتقدمة</span>
                    <span className="font-bold">35%</span>
                </div>
                <div className="w-full bg-gray-100 rounded-full h-2">
                    <div className="bg-green-500 h-2 rounded-full" style={{ width: '35%' }}></div>
                </div>

                <div className="flex items-center justify-between">
                    <span className="flex items-center gap-2"><div className="w-3 h-3 rounded-full bg-yellow-500"></div> الباقة الأساسية</span>
                    <span className="font-bold">20%</span>
                </div>
                <div className="w-full bg-gray-100 rounded-full h-2">
                    <div className="bg-yellow-500 h-2 rounded-full" style={{ width: '20%' }}></div>
                </div>
            </div>
          </div>

          <div className="bg-white p-6 rounded-2xl shadow-sm border border-gray-100">
             <h3 className="text-lg font-bold text-gray-800 mb-4">أداء السيرفرات</h3>
             <div className="flex items-center justify-center h-48">
                 <div className="relative w-32 h-32 rounded-full border-8 border-gray-100 flex items-center justify-center animate-spin-slow">
                     <div className="absolute inset-0 rounded-full border-8 border-success border-t-transparent rotate-45"></div>
                     <span className="text-2xl font-bold text-gray-700 animate-none" style={{ animationDirection: 'reverse' }}>98%</span>
                 </div>
             </div>
             <p className="text-center text-sm text-gray-500 mt-2">وقت التشغيل (Uptime)</p>
             <style>{`
                @keyframes spin-slow {
                    from { transform: rotate(0deg); }
                    to { transform: rotate(360deg); }
                }
                .animate-spin-slow {
                    animation: spin-slow 10s linear infinite;
                }
             `}</style>
          </div>
      </div>
    </div>
  );
};
