
import React, { useState } from 'react';
import { useData } from '../../context/DataContext';
import { Modal } from '../../components/Modal';
import { useToast } from '../../context/ToastContext';
import { Plan } from '../../types';

export const AdminPlans: React.FC = () => {
  const { plans, updatePlan } = useData();
  const { addToast } = useToast();
  const [editingPlan, setEditingPlan] = useState<Plan | null>(null);
  const [editPrice, setEditPrice] = useState<number>(0);

  const openEditModal = (plan: Plan) => {
    setEditingPlan(plan);
    setEditPrice(plan.price);
  };

  const handleUpdate = () => {
    if (editingPlan) {
      updatePlan({ ...editingPlan, price: editPrice });
      addToast('تم تحديث الخطة بنجاح', 'success');
      setEditingPlan(null);
    }
  };

  return (
    <div className="space-y-8">
      <div className="text-center max-w-2xl mx-auto">
        <h2 className="text-3xl font-bold text-gray-800 mb-3">خطط الاشتراك</h2>
        <p className="text-gray-500">إدارة الباقات والأسعار المتاحة للأطباء والعيادات</p>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-3 gap-8 px-4">
        {plans.map((plan) => (
            <div key={plan.id} className={`bg-white rounded-2xl shadow-lg border p-8 flex flex-col relative ${plan.isPopular ? 'border-primary ring-4 ring-primary ring-opacity-10 scale-105' : 'border-gray-100'}`}>
                {plan.isPopular && (
                    <div className="absolute top-0 right-1/2 transform translate-x-1/2 -translate-y-1/2 bg-primary text-white px-4 py-1 rounded-full text-sm font-bold shadow-md">
                        الأكثر طلباً
                    </div>
                )}
                <h3 className="text-xl font-bold text-gray-800 mb-2">{plan.name}</h3>
                <div className="text-4xl font-extrabold text-gray-900 mb-1">
                    ${plan.price} <span className="text-sm font-normal text-gray-500">/ {plan.period === 'monthly' ? 'شهري' : 'سنوي'}</span>
                </div>
                <p className="text-gray-500 text-sm mb-6">مناسب للأطباء المستقلين</p>
                
                <ul className="space-y-4 mb-8 flex-1">
                    {plan.features.map((feature, idx) => (
                        <li key={idx} className="flex items-center gap-3 text-gray-600">
                            <i className="fas fa-check-circle text-green-500"></i>
                            <span className="text-sm">{feature}</span>
                        </li>
                    ))}
                </ul>

                <button 
                  onClick={() => openEditModal(plan)}
                  className={`w-full py-3 rounded-xl font-bold transition-colors ${plan.isPopular ? 'bg-primary text-white hover:bg-primary-dark' : 'bg-gray-100 text-gray-800 hover:bg-gray-200'}`}
                >
                    تعديل السعر
                </button>
            </div>
        ))}
      </div>

      {editingPlan && (
        <Modal isOpen={!!editingPlan} onClose={() => setEditingPlan(null)} title={`تعديل ${editingPlan.name}`}>
            <div className="space-y-4">
                <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">السعر الجديد ($)</label>
                    <input 
                      type="number" 
                      value={editPrice}
                      onChange={(e) => setEditPrice(Number(e.target.value))}
                      className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-primary focus:border-primary bg-white text-gray-900" 
                    />
                </div>
                <button 
                  onClick={handleUpdate}
                  className="w-full bg-primary text-white py-2.5 rounded-xl font-bold hover:bg-primary-dark transition-colors"
                >
                  حفظ التغييرات
                </button>
            </div>
        </Modal>
      )}
    </div>
  );
};
