
import React, { useState } from 'react';
import { useData } from '../../context/DataContext';

export const AuditLogs: React.FC = () => {
  const { auditLogs } = useData();
  const [searchTerm, setSearchTerm] = useState('');
  const [roleFilter, setRoleFilter] = useState('all');

  const filteredLogs = auditLogs.filter(log => {
      const matchesSearch = log.details.toLowerCase().includes(searchTerm.toLowerCase()) || log.user.toLowerCase().includes(searchTerm.toLowerCase());
      const matchesRole = roleFilter === 'all' || log.role === roleFilter;
      return matchesSearch && matchesRole;
  });

  return (
    <div className="space-y-6">
      <div className="flex justify-between items-center mb-6">
        <div>
          <h2 className="text-2xl font-bold text-gray-800">سجل النشاطات (Audit Logs)</h2>
          <p className="text-gray-500">مراقبة جميع العمليات والتغييرات في النظام</p>
        </div>
        <button className="bg-white border border-gray-300 text-gray-700 px-4 py-2 rounded-lg shadow-sm hover:bg-gray-50 transition-colors flex items-center gap-2">
            <i className="fas fa-download"></i> تصدير CSV
        </button>
      </div>

      <div className="bg-white rounded-2xl shadow-sm border border-gray-100 overflow-hidden">
          <div className="p-4 bg-gray-50 border-b border-gray-200 flex flex-col md:flex-row gap-4">
              <div className="relative flex-1">
                  <input 
                    type="text" 
                    placeholder="بحث في السجلات..." 
                    className="w-full pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent bg-white text-gray-900"
                    value={searchTerm}
                    onChange={e => setSearchTerm(e.target.value)}
                  />
                  <i className="fas fa-search absolute left-3 top-3 text-gray-400"></i>
              </div>
              <select 
                className="px-4 py-2 border border-gray-300 rounded-lg bg-white text-gray-900"
                value={roleFilter}
                onChange={e => setRoleFilter(e.target.value)}
              >
                  <option value="all">كل الأدوار</option>
                  <option value="SUPER_ADMIN">Admin</option>
                  <option value="DOCTOR">Doctor</option>
                  <option value="SYSTEM">System</option>
              </select>
          </div>

          <div className="overflow-x-auto">
              <table className="w-full text-sm text-left">
                  <thead className="bg-gray-100 text-gray-700 font-semibold">
                      <tr>
                          <th className="px-6 py-3 text-right">الوقت</th>
                          <th className="px-6 py-3 text-right">المستخدم</th>
                          <th className="px-6 py-3 text-right">الدور</th>
                          <th className="px-6 py-3 text-right">النشاط</th>
                          <th className="px-6 py-3 text-right">التفاصيل</th>
                          <th className="px-6 py-3 text-right">IP Address</th>
                      </tr>
                  </thead>
                  <tbody className="divide-y divide-gray-100">
                      {filteredLogs.map(log => (
                          <tr key={log.id} className="hover:bg-gray-50">
                              <td className="px-6 py-3 whitespace-nowrap text-gray-600" dir="ltr">{log.timestamp}</td>
                              <td className="px-6 py-3 font-medium text-gray-800">{log.user}</td>
                              <td className="px-6 py-3">
                                  <span className={`px-2 py-1 rounded text-xs font-bold ${
                                      log.role === 'SUPER_ADMIN' ? 'bg-purple-100 text-purple-700' :
                                      log.role === 'DOCTOR' ? 'bg-blue-100 text-blue-700' :
                                      'bg-gray-200 text-gray-600'
                                  }`}>
                                      {log.role}
                                  </span>
                              </td>
                              <td className="px-6 py-3 text-gray-800 font-semibold">{log.action}</td>
                              <td className="px-6 py-3 text-gray-600 max-w-xs truncate" title={log.details}>{log.details}</td>
                              <td className="px-6 py-3 text-gray-500 font-mono text-xs">{log.ip}</td>
                          </tr>
                      ))}
                      {filteredLogs.length === 0 && (
                          <tr><td colSpan={6} className="text-center py-8 text-gray-400">لا توجد سجلات مطابقة</td></tr>
                      )}
                  </tbody>
              </table>
          </div>
      </div>
    </div>
  );
};
