
import React, { useState, useEffect } from 'react';
import { useToast } from '../../context/ToastContext';
import { useData } from '../../context/DataContext';

export const AdminSettings: React.FC = () => {
  const { addToast } = useToast();
  const { appSettings, updateAppSettings } = useData();
  
  // Local state for form handling
  const [platformName, setPlatformName] = useState(appSettings.platformName);
  const [supportEmail, setSupportEmail] = useState(appSettings.supportEmail);
  const [backupEnabled, setBackupEnabled] = useState(appSettings.backupEnabled);
  const [maintenanceMode, setMaintenanceMode] = useState(appSettings.maintenanceMode);
  
  // Backup State
  const [isBackingUp, setIsBackingUp] = useState(false);
  const [lastBackup, setLastBackup] = useState<string>('لم يتم النسخ بعد');
  const [backupHistory, setBackupHistory] = useState<string[]>([
      '2024-05-20 02:00:00 - Auto Backup',
      '2024-05-19 02:00:00 - Auto Backup'
  ]);

  // Sync when context changes
  useEffect(() => {
    setPlatformName(appSettings.platformName);
    setSupportEmail(appSettings.supportEmail);
    setBackupEnabled(appSettings.backupEnabled);
    setMaintenanceMode(appSettings.maintenanceMode);
  }, [appSettings]);

  const handleSave = () => {
    updateAppSettings({
      platformName,
      supportEmail,
      backupEnabled,
      maintenanceMode
    });
    addToast('تم حفظ الإعدادات العامة وتحديث النظام بنجاح', 'success');
  };

  const handleManualBackup = () => {
      setIsBackingUp(true);
      addToast('جاري بدء النسخ الاحتياطي لجميع المواقع...', 'info');
      
      // Simulate backup process
      setTimeout(() => {
          setIsBackingUp(false);
          const dateStr = new Date().toLocaleString('en-GB');
          setLastBackup(dateStr);
          setBackupHistory(prev => [`${dateStr} - Manual Backup (All Sites)`, ...prev]);
          addToast('تم إنشاء نسخة احتياطية كاملة بنجاح', 'success');
      }, 2500);
  };

  return (
    <div className="max-w-4xl mx-auto space-y-6">
      <header className="mb-6">
        <h2 className="text-2xl font-bold text-gray-800">الإعدادات العامة للنظام</h2>
        <p className="text-gray-500">التحكم في إعدادات المنصة الأساسية والنسخ الاحتياطي</p>
      </header>
      
      {maintenanceMode && (
        <div className="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-xl flex items-center gap-3">
          <i className="fas fa-exclamation-triangle"></i>
          <div>
            <span className="font-bold">تنبيه:</span> وضع الصيانة مفعل حالياً. قد لا يتمكن المستخدمون من الوصول للنظام.
          </div>
        </div>
      )}

      <div className="grid gap-6">
        {/* General Settings */}
        <div className="bg-white rounded-2xl shadow-sm border border-gray-100 p-6">
            <h3 className="text-lg font-bold text-gray-800 mb-4 flex items-center gap-2">
                <i className="fas fa-globe text-primary"></i>
                إعدادات الموقع
            </h3>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">اسم المنصة</label>
                    <input 
                      type="text" 
                      value={platformName}
                      onChange={(e) => setPlatformName(e.target.value)}
                      className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent bg-white text-gray-900" 
                    />
                </div>
                <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">بريد الدعم الفني</label>
                    <input 
                      type="email" 
                      value={supportEmail}
                      onChange={(e) => setSupportEmail(e.target.value)}
                      className="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent bg-white text-gray-900" 
                      dir="ltr" 
                    />
                </div>
            </div>
        </div>

        {/* Security & Maintenance */}
        <div className="bg-white rounded-2xl shadow-sm border border-gray-100 p-6">
            <h3 className="text-lg font-bold text-gray-800 mb-4 flex items-center gap-2">
                <i className="fas fa-shield-alt text-success"></i>
                الأمان والصيانة
            </h3>
            <div className="space-y-4">
                <div className="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
                    <div>
                        <h4 className="font-bold text-gray-800">النسخ الاحتياطي التلقائي (يومي)</h4>
                        <p className="text-xs text-gray-500">إجراء نسخ يومي لقواعد البيانات في الساعة 2:00 ص</p>
                    </div>
                    <label className="relative inline-flex items-center cursor-pointer">
                        <input 
                          type="checkbox" 
                          className="sr-only peer" 
                          checked={backupEnabled}
                          onChange={(e) => setBackupEnabled(e.target.checked)}
                        />
                        <div className="w-11 h-6 bg-gray-200 peer-focus:outline-none rounded-full peer peer-checked:after:translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-success"></div>
                    </label>
                </div>
                <div className="flex items-center justify-between p-4 bg-gray-50 rounded-lg border border-transparent hover:border-red-200 transition-colors">
                    <div>
                        <h4 className="font-bold text-gray-800">وضع الصيانة</h4>
                        <p className="text-xs text-gray-500">إيقاف النظام لجميع المستخدمين مؤقتاً للتحديثات</p>
                    </div>
                    <label className="relative inline-flex items-center cursor-pointer">
                        <input 
                          type="checkbox" 
                          className="sr-only peer" 
                          checked={maintenanceMode}
                          onChange={(e) => setMaintenanceMode(e.target.checked)}
                        />
                        <div className="w-11 h-6 bg-gray-200 peer-focus:outline-none rounded-full peer peer-checked:after:translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-red-500"></div>
                    </label>
                </div>
            </div>
        </div>

        {/* Full Backup Section */}
        <div className="bg-white rounded-2xl shadow-sm border border-gray-100 p-6">
            <div className="flex justify-between items-start mb-6">
                <div>
                    <h3 className="text-lg font-bold text-gray-800 flex items-center gap-2">
                        <i className="fas fa-database text-blue-600"></i>
                        النسخ الاحتياطي اليدوي
                    </h3>
                    <p className="text-sm text-gray-500 mt-1">إنشاء نسخة احتياطية فورية لجميع المواقع والبيانات (Downloadable SQL Dump)</p>
                </div>
                <button 
                    onClick={handleManualBackup}
                    disabled={isBackingUp}
                    className={`px-5 py-2.5 rounded-xl font-bold text-white shadow-md flex items-center gap-2 transition-all ${isBackingUp ? 'bg-gray-400 cursor-not-allowed' : 'bg-blue-600 hover:bg-blue-700'}`}
                >
                    {isBackingUp ? (
                        <>
                            <i className="fas fa-spinner fa-spin"></i> جاري النسخ...
                        </>
                    ) : (
                        <>
                            <i className="fas fa-cloud-download-alt"></i> نسخ احتياطي الآن
                        </>
                    )}
                </button>
            </div>

            <div className="bg-gray-50 rounded-xl p-4 border border-gray-200">
                <h4 className="font-bold text-gray-700 mb-3 text-sm border-b border-gray-200 pb-2">سجل النسخ الاحتياطي (آخر عمليات)</h4>
                <ul className="space-y-2 text-sm">
                    {backupHistory.map((log, idx) => (
                        <li key={idx} className="flex items-center gap-2 text-gray-600">
                            <i className="fas fa-check-circle text-green-500 text-xs"></i>
                            <span className="font-mono">{log}</span>
                        </li>
                    ))}
                </ul>
            </div>
        </div>

        <div className="flex justify-end pt-4 border-t border-gray-200">
            <button 
              onClick={handleSave}
              className="bg-primary text-white px-8 py-3 rounded-xl font-bold shadow-lg hover:bg-primary-dark transition-colors flex items-center gap-2"
            >
                <i className="fas fa-save"></i>
                حفظ الإعدادات
            </button>
        </div>
      </div>
    </div>
  );
};
