
import React, { useState, useRef, useEffect } from 'react';
import { GoogleGenAI, Chat } from "@google/genai";

interface Message {
  id: number;
  text: string;
  isBot: boolean;
}

export const ChatWidget: React.FC = () => {
  const [isOpen, setIsOpen] = useState(false);
  const [messages, setMessages] = useState<Message[]>([
    { id: 1, text: 'مرحباً بك! أنا مساعدك الذكي في ClinicPro. كيف يمكنني مساعدتك اليوم؟ 🤖', isBot: true }
  ]);
  const [inputText, setInputText] = useState('');
  const [isTyping, setIsTyping] = useState(false);
  const messagesEndRef = useRef<HTMLDivElement>(null);
  
  // Gemini Chat Session State
  const [chatSession, setChatSession] = useState<Chat | null>(null);

  const scrollToBottom = () => {
    messagesEndRef.current?.scrollIntoView({ behavior: "smooth" });
  };

  useEffect(() => {
    scrollToBottom();
  }, [messages, isOpen]);

  // Initialize Gemini Chat
  useEffect(() => {
    const initChat = async () => {
        try {
            if (process.env.API_KEY) {
                const ai = new GoogleGenAI({ apiKey: process.env.API_KEY });
                const chat = ai.chats.create({
                    model: 'gemini-2.5-flash',
                    config: {
                        systemInstruction: `أنت مساعد دعم فني ذكي لنظام "ClinicPro SaaS". 
                        دورك هو مساعدة الأطباء ومديري العيادات في استخدام النظام.
                        النظام يحتوي على إدارة المرضى، المواعيد، الفواتير، المخزون، والتقارير.
                        كن ودوداً، مختصراً، وتحدث باللغة العربية دائماً.
                        إذا سُئلت عن الأسعار، أجب بأن الباقة الشهرية تبدأ من 30$ والسنوية 300$.
                        `
                    }
                });
                setChatSession(chat);
            }
        } catch (error) {
            console.error("Failed to init AI chat", error);
        }
    };
    initChat();
  }, []);

  const handleSendMessage = async () => {
    if (!inputText.trim()) return;

    const userText = inputText;
    const newUserMsg: Message = { id: Date.now(), text: userText, isBot: false };
    setMessages(prev => [...prev, newUserMsg]);
    setInputText('');
    setIsTyping(true);

    try {
        if (chatSession) {
            // Use Real AI
            const response = await chatSession.sendMessage({ message: userText });
            const botText = response.text || 'عذراً، لم أستطع فهم ذلك.';
            
            setMessages(prev => [...prev, { id: Date.now() + 1, text: botText, isBot: true }]);
        } else {
            // Fallback Mock Logic (Offline or No Key)
            setTimeout(() => {
                let botResponseText = 'شكراً لتواصلك معنا. سيقوم أحد موظفي الدعم بالرد عليك قريباً.';
                if (userText.includes('سعر') || userText.includes('اشتراك')) {
                    botResponseText = 'يمكنك الاطلاع على خطط الأسعار في صفحة الاشتراكات. تبدأ من 30$ شهرياً.';
                }
                setMessages(prev => [...prev, { id: Date.now() + 1, text: botResponseText, isBot: true }]);
            }, 1000);
        }
    } catch (error) {
        setMessages(prev => [...prev, { id: Date.now() + 1, text: 'عذراً، حدث خطأ في الاتصال بالخادم.', isBot: true }]);
    } finally {
        setIsTyping(false);
    }
  };

  const handleKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === 'Enter') {
      handleSendMessage();
    }
  };

  return (
    <div className="fixed bottom-6 left-6 z-50 flex flex-col items-end">
      {/* Chat Window */}
      {isOpen && (
        <div className="mb-4 w-80 md:w-96 bg-white rounded-2xl shadow-2xl border border-gray-100 overflow-hidden flex flex-col transition-all transform origin-bottom-left animate-slide-up h-[500px]">
          <div className="bg-gradient-to-r from-primary to-primary-dark p-4 flex justify-between items-center text-white shadow-sm shrink-0">
            <div className="flex items-center gap-2">
                <div className="w-8 h-8 bg-white/20 rounded-full flex items-center justify-center">
                    <i className="fas fa-robot text-sm"></i>
                </div>
                <div>
                    <h4 className="font-bold text-sm">المساعد الذكي</h4>
                    <span className="text-[10px] text-white/80 flex items-center gap-1">
                        <span className="w-1.5 h-1.5 bg-green-400 rounded-full animate-pulse"></span> متصل الآن
                    </span>
                </div>
            </div>
            <button onClick={() => setIsOpen(false)} className="hover:text-gray-200 transition-colors">
              <i className="fas fa-times"></i>
            </button>
          </div>
          
          <div className="flex-1 p-4 overflow-y-auto bg-gray-50 flex flex-col gap-3">
            {messages.map((msg) => (
                <div 
                    key={msg.id} 
                    className={`p-3 rounded-2xl shadow-sm text-sm max-w-[85%] leading-relaxed ${
                        msg.isBot 
                        ? 'self-start bg-white border border-gray-100 text-gray-700 rounded-tl-none' 
                        : 'self-end bg-primary text-white rounded-tr-none'
                    }`}
                >
                    {msg.text}
                </div>
            ))}
            {isTyping && (
                <div className="self-start bg-white border border-gray-100 p-3 rounded-2xl rounded-tl-none shadow-sm w-16 flex items-center justify-center gap-1">
                    <div className="w-2 h-2 bg-gray-400 rounded-full animate-bounce"></div>
                    <div className="w-2 h-2 bg-gray-400 rounded-full animate-bounce delay-100"></div>
                    <div className="w-2 h-2 bg-gray-400 rounded-full animate-bounce delay-200"></div>
                </div>
            )}
            <div ref={messagesEndRef} />
          </div>

          <div className="p-3 border-t border-gray-100 bg-white flex gap-2 shrink-0">
            <input 
              type="text" 
              placeholder="اكتب رسالتك..." 
              className="flex-1 px-4 py-2.5 bg-gray-50 border border-gray-200 rounded-xl text-sm focus:outline-none focus:border-primary focus:ring-1 focus:ring-primary transition-all text-gray-900"
              value={inputText}
              onChange={(e) => setInputText(e.target.value)}
              onKeyDown={handleKeyPress}
            />
            <button 
                onClick={handleSendMessage}
                disabled={!inputText.trim() || isTyping}
                className="w-10 h-10 bg-primary text-white rounded-xl hover:bg-primary-dark transition-colors flex items-center justify-center disabled:opacity-50 disabled:cursor-not-allowed shadow-md"
            >
              <i className="fas fa-paper-plane"></i>
            </button>
          </div>
        </div>
      )}

      {/* Floating Button */}
      <button 
        onClick={() => setIsOpen(!isOpen)}
        className="w-14 h-14 bg-gradient-to-br from-primary to-blue-600 hover:from-primary-dark hover:to-blue-700 text-white rounded-full shadow-xl flex items-center justify-center transition-all hover:scale-110 border-2 border-white"
      >
        <i className={`fas ${isOpen ? 'fa-times' : 'fa-headset'} text-2xl`}></i>
      </button>
    </div>
  );
};
