
import React, { useState, useEffect, useRef } from 'react';
import { HashRouter, Routes, Route, Navigate, useNavigate, useLocation } from 'react-router-dom';
import { Sidebar } from './components/Sidebar';
import { Landing } from './pages/Landing';
import { ChatWidget } from './components/ChatWidget';
import { ToastProvider } from './context/ToastContext';
import { DataProvider, useData } from './context/DataContext';

// Admin Pages
import { AdminDashboard } from './pages/admin/AdminDashboard';
import { AdminTenants } from './pages/admin/Tenants';
import { AdminPlans } from './pages/admin/Plans';
import { AdminTemplates } from './pages/admin/Templates';
import { AdminSettings } from './pages/admin/AdminSettings';
import { AdminReports } from './pages/admin/Reports';
import { FormBuilder } from './pages/admin/FormBuilder';
import { PageManager } from './pages/admin/PageManager';
import { AuditLogs } from './pages/admin/AuditLogs';
import { SupportCenter } from './pages/admin/SupportCenter';
import { RoleManager } from './pages/admin/RoleManager'; // New

// Doctor Pages
import { DoctorDashboard } from './pages/doctor/DoctorDashboard';
import { PatientList } from './pages/doctor/PatientList';
import { PatientDetails } from './pages/doctor/PatientDetails';
import { Appointments } from './pages/doctor/Appointments';
import { Finance } from './pages/doctor/Finance';
import { Services } from './pages/doctor/Services';
import { DoctorSettings } from './pages/doctor/Settings';
import { DentalLab } from './pages/doctor/DentalLab';
import { BirthLog } from './pages/doctor/BirthLog';
import { Messages } from './pages/doctor/Messages';
import { DynamicPage } from './pages/doctor/DynamicPage';
import { Inventory } from './pages/doctor/Inventory';
import { Prescriptions } from './pages/doctor/Prescriptions';
import { Staff } from './pages/doctor/Staff';
import { MedicalReports } from './pages/doctor/MedicalReports';
import { SiteEditor } from './pages/doctor/SiteEditor';
import { FieldManager } from './pages/doctor/FieldManager'; // New

// Public Pages
import { PatientWebsite } from './pages/public/PatientWebsite';
import { WaitingRoom } from './pages/public/WaitingRoom';

import { UserRole } from './types';

// Wrapper component to handle layout based on authentication state
const AppLayout: React.FC<{ role: UserRole | null; onLogout: () => void }> = ({ role, onLogout }) => {
  const [sidebarOpen, setSidebarOpen] = useState(false);
  const [notificationsOpen, setNotificationsOpen] = useState(false);
  const navigate = useNavigate();
  const location = useLocation();
  const notificationRef = useRef<HTMLDivElement>(null);

  // Consume Data
  const { notifications, markNotificationAsRead, clearNotifications, appSettings } = useData();
  const unreadCount = notifications.filter(n => !n.read).length;

  // Close notifications on click outside
  useEffect(() => {
    const handleClickOutside = (event: MouseEvent) => {
      if (notificationRef.current && !notificationRef.current.contains(event.target as Node)) {
        setNotificationsOpen(false);
      }
    };
    document.addEventListener('mousedown', handleClickOutside);
    return () => document.removeEventListener('mousedown', handleClickOutside);
  }, []);

  // Determine active page ID based on path
  const getActivePage = () => {
    const path = location.pathname;
    if (path.includes('admin/dashboard')) return 'admin-dashboard';
    if (path.includes('admin/reports')) return 'admin-reports';
    if (path.includes('admin/tenants')) return 'admin-tenants';
    if (path.includes('admin/plans')) return 'admin-plans';
    if (path.includes('admin/templates')) return 'admin-templates';
    if (path.includes('admin/form-builder')) return 'admin-form-builder';
    if (path.includes('admin/features')) return 'admin-features';
    if (path.includes('admin/audit')) return 'admin-audit';
    if (path.includes('admin/support')) return 'admin-support';
    if (path.includes('admin/roles')) return 'admin-roles';
    if (path.includes('admin/settings')) return 'admin-settings';
    
    if (path.includes('doctor/dashboard')) return 'doctor-dashboard';
    if (path.includes('doctor/patients')) return 'doctor-patients';
    if (path.includes('doctor/appointments')) return 'doctor-appointments';
    if (path.includes('doctor/finance')) return 'doctor-finance';
    if (path.includes('doctor/services')) return 'doctor-services';
    if (path.includes('doctor/settings')) return 'doctor-settings';
    if (path.includes('doctor/messages')) return 'doctor-messages';
    
    if (path.includes('doctor/dental-lab')) return 'dental-lab';
    if (path.includes('doctor/birth-log')) return 'birth-log';
    if (path.includes('doctor/inventory')) return 'doctor-inventory';
    if (path.includes('doctor/prescriptions')) return 'doctor-prescriptions';
    if (path.includes('doctor/staff')) return 'doctor-staff';
    if (path.includes('doctor/reports')) return 'doctor-reports';
    if (path.includes('doctor/site-editor')) return 'doctor-site-editor';
    if (path.includes('doctor/forms')) return 'doctor-forms';
    if (path.includes('doctor/field-manager')) return 'doctor-field-manager';

    // Check dynamic pages
    if (path.includes('doctor/page-')) return path.split('/doctor/')[1];

    return '';
  };

  const handleNavigation = (pageId: string) => {
    if (pageId.startsWith('page-')) {
        navigate(`/doctor/${pageId}`);
        return;
    }

    switch (pageId) {
        case 'admin-dashboard': navigate('/admin/dashboard'); break;
        case 'admin-reports': navigate('/admin/reports'); break;
        case 'admin-tenants': navigate('/admin/tenants'); break;
        case 'admin-plans': navigate('/admin/plans'); break;
        case 'admin-templates': navigate('/admin/templates'); break;
        case 'admin-form-builder': navigate('/admin/form-builder'); break;
        case 'admin-features': navigate('/admin/features'); break;
        case 'admin-audit': navigate('/admin/audit'); break;
        case 'admin-support': navigate('/admin/support'); break;
        case 'admin-roles': navigate('/admin/roles'); break;
        case 'admin-settings': navigate('/admin/settings'); break;
        
        case 'doctor-dashboard': navigate('/doctor/dashboard'); break;
        case 'doctor-patients': navigate('/doctor/patients'); break;
        case 'doctor-appointments': navigate('/doctor/appointments'); break;
        case 'doctor-finance': navigate('/doctor/finance'); break;
        case 'doctor-services': navigate('/doctor/services'); break;
        case 'doctor-settings': navigate('/doctor/settings'); break;
        case 'doctor-messages': navigate('/doctor/messages'); break;
        case 'dental-lab': navigate('/doctor/dental-lab'); break;
        case 'birth-log': navigate('/doctor/birth-log'); break;
        case 'doctor-inventory': navigate('/doctor/inventory'); break;
        case 'doctor-prescriptions': navigate('/doctor/prescriptions'); break;
        case 'doctor-staff': navigate('/doctor/staff'); break;
        case 'doctor-reports': navigate('/doctor/reports'); break;
        case 'doctor-site-editor': navigate('/doctor/site-editor'); break;
        case 'doctor-forms': navigate('/doctor/forms'); break;
        case 'doctor-field-manager': navigate('/doctor/field-manager'); break;
        default: break;
    }
  };

  if (!role) return <Navigate to="/" replace />;

  // Maintenance Mode Check
  if (appSettings.maintenanceMode && role !== UserRole.SUPER_ADMIN) {
      return (
          <div className="h-screen flex flex-col items-center justify-center bg-gray-50 text-center p-6">
              <i className="fas fa-tools text-6xl text-gray-300 mb-6"></i>
              <h1 className="text-3xl font-bold text-gray-800 mb-2">النظام تحت الصيانة</h1>
              <p className="text-gray-500 max-w-md mx-auto">
                  نقوم حالياً بإجراء بعض التحسينات على النظام لتقديم خدمة أفضل. نعتذر عن الإزعاج ونعود إليكم قريباً.
              </p>
              <button 
                onClick={onLogout}
                className="mt-8 text-primary font-bold hover:underline flex items-center gap-2"
              >
                  <i className="fas fa-arrow-right"></i> تسجيل الخروج
              </button>
          </div>
      );
  }

  return (
    <div className="flex h-screen bg-gray-50 overflow-hidden">
      {/* Sidebar */}
      <Sidebar 
        role={role} 
        isOpen={sidebarOpen} 
        activePage={getActivePage()}
        onNavigate={handleNavigation}
        onLogout={onLogout}
        onCloseMobile={() => setSidebarOpen(false)}
      />

      {/* Main Content */}
      <div className="flex-1 flex flex-col h-full overflow-hidden relative">
        {/* Header */}
        <header className="h-16 bg-white shadow-sm flex items-center justify-between px-6 z-10">
          <button 
            className="lg:hidden text-gray-500 hover:text-primary"
            onClick={() => setSidebarOpen(true)}
          >
            <i className="fas fa-bars text-xl"></i>
          </button>
          
          <div className="flex-1"></div>

          <div className="flex items-center gap-4">
             {/* External Link for Waiting Room */}
             {role === UserRole.DOCTOR && (
               <a href="/#/waiting-room" target="_blank" className="hidden md:flex items-center gap-2 bg-gray-100 hover:bg-gray-200 text-gray-700 px-3 py-1.5 rounded-lg text-sm transition-colors" title="فتح شاشة الانتظار">
                 <i className="fas fa-tv"></i> شاشة الانتظار
               </a>
             )}

             {/* Notification Bell */}
            <div className="relative" ref={notificationRef}>
              <button 
                onClick={() => setNotificationsOpen(!notificationsOpen)}
                className="relative p-2 text-gray-400 hover:text-primary transition-colors focus:outline-none"
              >
                <i className={`fas fa-bell text-xl ${unreadCount > 0 ? 'text-primary' : ''}`}></i>
                {unreadCount > 0 && (
                  <span className="absolute top-1 right-1 w-2.5 h-2.5 bg-red-500 rounded-full border-2 border-white animate-pulse"></span>
                )}
              </button>
              
              {notificationsOpen && (
                <div className="absolute left-0 mt-3 w-80 bg-white rounded-xl shadow-lg border border-gray-100 py-2 z-50 transform origin-top-left">
                  <div className="px-4 py-2 border-b border-gray-100 flex justify-between items-center bg-gray-50">
                    <h3 className="font-bold text-gray-800 text-sm">التنبيهات ({unreadCount})</h3>
                    {notifications.length > 0 && (
                      <button onClick={clearNotifications} className="text-xs text-red-500 font-medium hover:underline">مسح الكل</button>
                    )}
                  </div>
                  <div className="max-h-72 overflow-y-auto">
                    {notifications.length === 0 ? (
                      <div className="text-center py-6 text-gray-400 text-sm">لا توجد تنبيهات جديدة</div>
                    ) : (
                      notifications.map((notif) => (
                        <div 
                          key={notif.id} 
                          onClick={() => markNotificationAsRead(notif.id)}
                          className={`px-4 py-3 hover:bg-gray-50 transition-colors border-b border-gray-50 last:border-0 cursor-pointer ${notif.read ? 'opacity-60' : 'bg-blue-50/30'}`}
                        >
                          <div className="flex gap-3">
                            <div className={`w-8 h-8 rounded-full flex items-center justify-center shrink-0 ${
                                notif.type === 'success' ? 'bg-green-100 text-green-600' :
                                notif.type === 'warning' ? 'bg-yellow-100 text-yellow-600' :
                                notif.type === 'error' ? 'bg-red-100 text-red-600' :
                                'bg-blue-100 text-blue-600'
                            }`}>
                              <i className={`fas ${
                                notif.type === 'success' ? 'fa-check' :
                                notif.type === 'warning' ? 'fa-exclamation' :
                                notif.type === 'error' ? 'fa-times' :
                                'fa-info'
                              } text-sm`}></i>
                            </div>
                            <div>
                              <p className={`text-sm text-gray-800 ${!notif.read ? 'font-bold' : 'font-medium'}`}>
                                {notif.title}
                              </p>
                              <p className="text-xs text-gray-600 mt-0.5">{notif.message}</p>
                              <p className="text-[10px] text-gray-400 mt-1">{notif.timestamp}</p>
                            </div>
                          </div>
                        </div>
                      ))
                    )}
                  </div>
                </div>
              )}
            </div>
            
            {/* User Profile */}
            <div className="flex items-center gap-3 pl-2 cursor-pointer hover:bg-gray-50 p-1.5 rounded-lg transition-colors">
              <div className="text-right hidden md:block">
                <p className="text-sm font-bold text-gray-800">
                  {role === UserRole.SUPER_ADMIN ? 'المدير العام' : 'د. أحمد خالد'}
                </p>
                <p className="text-xs text-gray-500">
                  {role === UserRole.SUPER_ADMIN ? 'Super Admin' : 'عيادة الشفاء'}
                </p>
              </div>
              <div className="w-10 h-10 rounded-full bg-gray-200 border-2 border-white shadow-sm overflow-hidden">
                 <img src="https://picsum.photos/id/64/200/200" alt="Avatar" className="w-full h-full object-cover" />
              </div>
            </div>
          </div>
        </header>

        {/* Page Content */}
        <main className="flex-1 overflow-x-hidden overflow-y-auto p-4 md:p-6">
          <Routes>
            {/* Admin Routes */}
            <Route path="/admin/dashboard" element={<AdminDashboard />} />
            <Route path="/admin/reports" element={<AdminReports />} />
            <Route path="/admin/tenants" element={<AdminTenants />} />
            <Route path="/admin/plans" element={<AdminPlans />} />
            <Route path="/admin/templates" element={<AdminTemplates />} />
            <Route path="/admin/form-builder" element={<FormBuilder />} />
            <Route path="/admin/features" element={<PageManager />} />
            <Route path="/admin/audit" element={<AuditLogs />} />
            <Route path="/admin/support" element={<SupportCenter />} />
            <Route path="/admin/roles" element={<RoleManager />} />
            <Route path="/admin/settings" element={<AdminSettings />} />
            
            {/* Doctor Routes */}
            <Route path="/doctor/dashboard" element={<DoctorDashboard />} />
            <Route path="/doctor/patients/:id" element={<PatientDetails />} />
            <Route path="/doctor/patients" element={<PatientList />} />
            <Route path="/doctor/appointments" element={<Appointments />} />
            <Route path="/doctor/finance" element={<Finance />} />
            <Route path="/doctor/services" element={<Services />} />
            <Route path="/doctor/settings" element={<DoctorSettings />} />
            <Route path="/doctor/messages" element={<Messages />} />
            <Route path="/doctor/dental-lab" element={<DentalLab />} />
            <Route path="/doctor/birth-log" element={<BirthLog />} />
            <Route path="/doctor/inventory" element={<Inventory />} />
            <Route path="/doctor/prescriptions" element={<Prescriptions />} />
            <Route path="/doctor/staff" element={<Staff />} />
            <Route path="/doctor/reports" element={<MedicalReports />} />
            <Route path="/doctor/site-editor" element={<SiteEditor />} />
            <Route path="/doctor/forms" element={<FormBuilder />} />
            <Route path="/doctor/field-manager" element={<FieldManager />} />
            
            {/* Dynamic Doctor Pages */}
            <Route path="/doctor/page-:id" element={<DynamicPage />} />
            
            {/* Fallback */}
            <Route path="*" element={<Navigate to={role === UserRole.SUPER_ADMIN ? "/admin/dashboard" : "/doctor/dashboard"} replace />} />
          </Routes>
        </main>

        {/* Support Chat Widget */}
        <ChatWidget />
      </div>
    </div>
  );
};

const App: React.FC = () => {
  const [userRole, setUserRole] = useState<UserRole | null>(null);

  const handleLogin = (role: UserRole) => {
    setUserRole(role);
  };

  const handleLogout = () => {
    setUserRole(null);
  };

  return (
    <DataProvider>
      <ToastProvider>
        <HashRouter>
          <Routes>
            {/* Public Website Route - Needs DataProvider */}
            <Route path="/site/:domain" element={<PatientWebsite />} />
            
            {/* Waiting Room Route - Needs DataProvider */}
            <Route path="/waiting-room" element={<WaitingRoom />} />

            {/* Main App Routes */}
            <Route path="/*" element={
              !userRole ? (
                <Landing onLogin={handleLogin} />
              ) : (
                <AppLayout role={userRole} onLogout={handleLogout} />
              )
            } />
          </Routes>
        </HashRouter>
      </ToastProvider>
    </DataProvider>
  );
};

export default App;
