
export enum UserRole {
  SUPER_ADMIN = 'SUPER_ADMIN',
  DOCTOR = 'DOCTOR',
  NURSE = 'NURSE',
  RECEPTIONIST = 'RECEPTIONIST',
  GUEST = 'GUEST'
}

// 1. Expanded Clinic Types
export type ClinicType = 
  | 'general' 
  | 'dental' 
  | 'obgyn' 
  | 'vet'        // بيطري
  | 'ophthalmology' // عيون
  | 'orthopedics'   // عظام
  | 'surgery'       // جراحة
  | 'pediatrics'    // أطفال
  | 'neurology'     // أعصاب
  | 'dermatology'   // جلدية
  | 'obstetrics';   // توليد

export enum SubscriptionStatus {
  ACTIVE = 'ACTIVE',
  EXPIRED = 'EXPIRED',
  PENDING = 'PENDING'
}

// --- Sync System ---
export interface SyncAction {
  id: string;
  type: 'CREATE' | 'UPDATE' | 'DELETE';
  collection: string; // e.g., 'patients', 'appointments'
  payload: any;
  timestamp: number;
}

export interface AppSettings {
  platformName: string;
  supportEmail: string;
  backupEnabled: boolean;
  maintenanceMode: boolean;
}

export interface Notification {
  id: number;
  title: string;
  message: string;
  type: 'info' | 'warning' | 'success' | 'error';
  timestamp: string;
  read: boolean;
}

// CMS Content Structure
export interface SiteContent {
  heroTitle: string;
  heroSubtitle: string;
  aboutTitle: string;
  aboutText: string;
  contactAddress: string;
  contactPhone: string;
  contactEmail: string;
  primaryColor?: string; // Optional custom color override
}

export interface Tenant {
  id: string;
  name: string;
  email: string;
  phone: string;
  clinicName: string;
  clinicType: ClinicType; // Added type here
  plan: 'monthly' | 'yearly';
  status: SubscriptionStatus;
  expiryDate: string;
  domain: string;
  template: string;
  activationCode: string;
  isActivated: boolean;
  syncStatus: 'online' | 'offline'; // New: Sync Status
  siteContent?: SiteContent; // New: CMS Data
}

// --- Dynamic Configuration System (New) ---

export interface FieldConfig {
  id: string; // Unique key (e.g., 'personal_name', 'history_diabetes')
  label: string; // Display label (Editable)
  type: 'text' | 'textarea' | 'number' | 'date' | 'select' | 'boolean';
  section: string; // 'personal', 'history', 'obgyn', etc.
  isEnabled: boolean; // Master toggle
  isVisibleToPatient: boolean; // Public/Private toggle
  options?: string[]; // For select inputs
  width?: 'full' | 'half';
  order: number;
  isSystem?: boolean; // Prevent deletion of core fields
}

// --- Medical Records Interfaces ---

export interface MedicalHistory {
  chronicDiseases: string[];
  currentMedications: string[];
  allergies: string[];
  pastSurgeries: string[];
  familyHistory: string;
  bloodType: string;
}

export interface VitalSigns {
  date: string;
  temperature: number;
  bloodPressure: string;
  heartRate: number;
  weight: number;
  height: number;
  bmi: number;
}

export interface ObgynRecord {
  pregnancyWeeks: number;
  fetusWeight: number;
  fetusCount: number;
  pregnancyHistory: string;
  complications: string;
  medications: string;
  lastVisitDate: string;
  labResults: string[];
}

export interface PediatricRecord {
  birthDate: string;
  birthWeight: number;
  birthHeight: number;
  vaccinations: string[];
  developmentMilestones: string;
  geneticDiseases: string;
}

export interface SurgeryRecord {
  surgeryName: string;
  surgeryDate: string;
  surgeryType: string;
  postOpInstructions: string;
  recoveryPeriod: string;
  medications: string;
  complications: string;
}

export interface DentalRecord {
  teethStatus: Record<number, 'healthy' | 'decay' | 'treated' | 'missing'>;
  treatmentType: string;
  treatedTeethCount: number;
  pastCheckups: string;
  medications: string;
  symptoms: string;
}

export interface ConsultationRecord {
  id: string;
  date: string;
  subject: string;
  notes: string;
  doctorReply: string;
  recommendations: string;
  attachments: Attachment[];
}

export interface Attachment {
  id: string;
  name: string;
  url: string;
  type: 'image' | 'pdf' | 'lab';
  date: string;
}

// --- Dynamic System Interfaces (No-Code Core) ---

export type FieldType = 
  // Inputs
  | 'text' 
  | 'textarea' 
  | 'number' 
  | 'email' 
  | 'phone_input' 
  | 'address'     
  | 'date' 
  | 'time' 
  // Selectors
  | 'select' 
  | 'checkbox' 
  | 'radio' 
  | 'multiselect' 
  | 'switch'      
  | 'slider'      
  | 'rating'
  // Advanced & Media
  | 'file'
  | 'signature'   
  | 'static_image'
  | 'interactive_image' 
  | 'chart'             
  | 'video'        
  | 'map'          
  | 'html_block'    
  | 'countdown'     
  | 'accordion'     
  | 'social_share'  
  | 'progress_bar'  
  // Layout & Interactive
  | 'header'      
  | 'paragraph'   
  | 'divider'     
  | 'page_break'   
  | 'cta_button'   
  | 'navigation'   
  | 'alert_box';   

export interface FormField {
  id: string;
  type: FieldType;
  label: string;
  placeholder?: string;
  required: boolean;
  options?: string[]; // For select, radio, multiselect, chart labels, accordion items
  width?: 'full' | 'half' | 'third'; // Layout control
  defaultValue?: string; // Used for button links, video URLs, or alert types
  animation?: 'none' | 'fade-in' | 'slide-up' | 'zoom-in' | 'bounce'; 
}

export interface CustomForm {
  id: string;
  title: string;
  description?: string;
  targetLocation?: string; // New: Where to display the form (Menu, Sidebar, Page)
  clinicType: ClinicType | 'all'; // Target audience
  fields: FormField[];
  isSystem: boolean; // Cannot be deleted if true
  createdAt: string;
}

export interface ModuleFeature {
  id: string;
  key: string; 
  defaultName: string; 
  customName?: string; 
  isEnabled: boolean;
  icon: string;
  clinicTypes: ClinicType[]; 
  description?: string;
}

export interface DynamicPage {
  id: string;
  title: string;
  path: string; // URL path slug
  icon: string;
  type: 'form' | 'dashboard' | 'report';
  connectedFormId?: string; // If type is form, which form schema to load
  clinicType: ClinicType | 'all';
  order: number;
  isVisible: boolean;
}

// --- Existing Data Models ---

export interface Patient {
  id: string;
  name: string;
  age: number;
  phone: string;
  email?: string;
  address?: string;
  city?: string;
  gender?: 'male' | 'female';
  maritalStatus?: 'single' | 'married';
  profession?: string;
  nationalId?: string;
  diagnosis: string;
  lastVisit: string;
  status: 'pregnant' | 'regular' | 'critical';
  
  // Linked Records
  medicalHistory?: MedicalHistory;
  obgynRecord?: ObgynRecord;
  pediatricRecord?: PediatricRecord;
  surgeryRecord?: SurgeryRecord;
  dentalRecord?: DentalRecord;
  consultations?: ConsultationRecord[];
  
  // Dynamic fields storage: This allows the "No-Code" fields to be saved
  customData?: Record<string, any>; 
}

export interface Visit {
  id: string;
  patientId: string;
  date: string;
  type: string;
  cost: number;
  paid: number;
  notes: string;
}

export interface DashboardStats {
  totalPatients: number;
  incomeToday: number;
  activeAppointments: number;
  alerts: number;
}

export interface Appointment {
  id: string;
  patientName: string;
  date: string;
  time: string;
  type: string;
  status: 'confirmed' | 'pending' | 'cancelled' | 'completed';
  notes?: string;
}

export interface Invoice {
  id: string;
  patientName: string;
  date: string;
  amount: number;
  status: 'paid' | 'pending' | 'overdue';
  service: string;
}

export interface Plan {
  id: string;
  name: string;
  price: number;
  period: 'monthly' | 'yearly';
  features: string[];
  isPopular?: boolean;
}

export interface Service {
  id: string;
  name: string;
  price: number;
  description?: string;
}

export interface Template {
  id: string;
  name: string;
  thumbnail: string;
  category: string;
  isActive: boolean;
}

export interface LabOrder {
  id: string;
  patientName: string;
  item: string;
  labName: string;
  sentDate: string;
  deliveryDate: string;
  status: 'sent' | 'in_progress' | 'received';
  shade: string;
}

export interface BirthRecord {
  id: string;
  motherName: string;
  babyName: string;
  date: string;
  time: string;
  gender: 'male' | 'female';
  weight: number;
  deliveryType: 'natural' | 'c-section';
  notes?: string;
}

// --- New Features Interfaces ---

export interface InventoryItem {
  id: string;
  name: string;
  quantity: number;
  unit: string;
  minThreshold: number;
  expiryDate?: string;
  category: 'medicine' | 'consumable' | 'equipment';
}

export interface PrescriptionItem {
  drugName: string;
  dosage: string;
  frequency: string;
  duration: string;
  notes?: string;
}

export interface Prescription {
  id: string;
  patientId: string;
  patientName: string;
  date: string;
  items: PrescriptionItem[];
  doctorName: string;
}

// Staff & Reports
export interface ClinicStaff {
  id: string;
  name: string;
  role: 'doctor' | 'nurse' | 'receptionist';
  email: string;
  phone: string;
  status: 'active' | 'inactive';
  permissions: string[]; // List of allowed actions/pages
  password?: string; // New: Password for staff login
}

export interface RolePermission {
  role: string;
  canEditRecords: boolean;
  canViewFinancials: boolean;
  canManageSettings: boolean;
  canDelete: boolean;
}

export interface MedicalReport {
  id: string;
  patientName: string;
  type: 'sick_leave' | 'referral' | 'general';
  date: string;
  content: string;
  doctorName: string;
}

// --- Enterprise Features ---

export interface AuditLog {
  id: string;
  action: string;
  user: string;
  role: string;
  details: string;
  timestamp: string;
  ip: string;
}

export interface SupportTicket {
  id: string;
  tenantId: string;
  tenantName: string;
  subject: string;
  message: string;
  status: 'open' | 'in_progress' | 'resolved';
  priority: 'low' | 'medium' | 'high';
  createdAt: string;
  lastUpdate: string;
}
