
import React, { useState } from 'react';
import { useData } from '../../context/DataContext';
import { Modal } from '../../components/Modal';
import { useToast } from '../../context/ToastContext';
import { ClinicStaff } from '../../types';

export const Staff: React.FC = () => {
  const { staff, addStaff, deleteStaff } = useData();
  const { addToast } = useToast();
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [editingStaff, setEditingStaff] = useState<ClinicStaff | null>(null);
  
  // Local state for permissions during edit/create
  const [selectedPermissions, setSelectedPermissions] = useState<string[]>([]);

  // Defined Permissions List
  const AVAILABLE_PERMISSIONS = [
      { id: 'patients_view', label: 'عرض سجل المرضى' },
      { id: 'patients_edit', label: 'تعديل/إضافة مرضى' },
      { id: 'appointments_manage', label: 'إدارة المواعيد' },
      { id: 'financials_view', label: 'الاطلاع على الحسابات' },
      { id: 'inventory_manage', label: 'إدارة المخزون' },
      { id: 'reports_view', label: 'عرض التقارير الطبية' },
      { id: 'settings_manage', label: 'تعديل إعدادات العيادة' },
      { id: 'prescriptions_create', label: 'كتابة الوصفات' },
      { id: 'field_manage', label: 'إدارة الحقول والنماذج (Field Manager)' }, // New Permission
  ];

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    const form = e.target as HTMLFormElement;
    const name = (form.elements.namedItem('name') as HTMLInputElement).value;
    const role = (form.elements.namedItem('role') as HTMLSelectElement).value as 'doctor' | 'nurse' | 'receptionist';
    const email = (form.elements.namedItem('email') as HTMLInputElement).value;
    const phone = (form.elements.namedItem('phone') as HTMLInputElement).value;
    const password = (form.elements.namedItem('password') as HTMLInputElement).value;
    const status = (form.elements.namedItem('status') as HTMLSelectElement).value as 'active' | 'inactive';

    if (name && role) {
        // If "all" was previously selected (admin), keep it or manage it carefully. 
        // For simplicity, if editing a doctor, we might default to 'all' or specific permissions.
        const permissionsToSave = role === 'doctor' ? ['all'] : selectedPermissions;

        if (editingStaff) {
            deleteStaff(editingStaff.id); // Remove old
            const updatedStaff: ClinicStaff = {
                id: editingStaff.id,
                name,
                role,
                email,
                phone,
                status,
                permissions: permissionsToSave,
                password: password || editingStaff.password // Update if new, else keep old
            };
            addStaff(updatedStaff);
            addToast('تم تحديث بيانات الموظف وصلاحياته', 'success');
        } else {
            // Add Mode
            const newStaff: ClinicStaff = {
                id: `s_${Date.now()}`,
                name,
                role,
                email,
                phone,
                status,
                permissions: permissionsToSave,
                password
            };
            addStaff(newStaff);
            addToast('تم إضافة الموظف بنجاح', 'success');
        }
        setIsModalOpen(false);
        setEditingStaff(null);
        setSelectedPermissions([]);
    }
  };

  const handleDelete = (id: string) => {
      if(window.confirm('هل أنت متأكد من حذف هذا الموظف؟')) {
          deleteStaff(id);
          addToast('تم حذف الموظف', 'error');
      }
  };

  const handleEdit = (member: ClinicStaff) => {
      setEditingStaff(member);
      setSelectedPermissions(member.permissions || []);
      setIsModalOpen(true);
  };

  const togglePermission = (permId: string) => {
      setSelectedPermissions(prev => 
          prev.includes(permId) ? prev.filter(p => p !== permId) : [...prev, permId]
      );
  };

  const getRoleLabel = (role: string) => {
      switch(role) {
          case 'doctor': return 'طبيب';
          case 'nurse': return 'ممرض/ـة';
          case 'receptionist': return 'استقبال';
          default: return role;
      }
  };

  return (
    <div className="space-y-6">
      <div className="flex justify-between items-center mb-6">
        <div>
          <h2 className="text-2xl font-bold text-gray-800">طاقم العمل والصلاحيات</h2>
          <p className="text-gray-500">إدارة الموظفين وتحديد الميزات المسموحة لكل منهم</p>
        </div>
        <button 
          onClick={() => { setEditingStaff(null); setSelectedPermissions([]); setIsModalOpen(true); }}
          className="bg-primary hover:bg-primary-dark text-white px-5 py-2.5 rounded-xl shadow-md transition-colors flex items-center gap-2"
        >
          <i className="fas fa-user-plus"></i>
          <span>إضافة موظف</span>
        </button>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {staff.map((member) => (
              <div key={member.id} className="bg-white p-6 rounded-2xl shadow-sm border border-gray-100 flex items-start gap-4 hover:shadow-md transition-shadow">
                  <div className={`w-14 h-14 rounded-full flex items-center justify-center text-2xl text-white ${
                      member.role === 'doctor' ? 'bg-blue-500' : 
                      member.role === 'nurse' ? 'bg-green-500' : 'bg-purple-500'
                  }`}>
                      <i className={`fas ${
                          member.role === 'doctor' ? 'fa-user-md' : 
                          member.role === 'nurse' ? 'fa-user-nurse' : 'fa-headset'
                      }`}></i>
                  </div>
                  <div className="flex-1">
                      <div className="flex justify-between items-start">
                          <h4 className="font-bold text-gray-800 text-lg">{member.name}</h4>
                          <span className={`text-[10px] px-2 py-0.5 rounded-full uppercase font-bold ${
                              member.status === 'active' ? 'bg-green-100 text-green-700' : 'bg-gray-100 text-gray-600'
                          }`}>{member.status === 'active' ? 'نشط' : 'غير نشط'}</span>
                      </div>
                      <p className="text-sm text-gray-500 mb-2">{getRoleLabel(member.role)}</p>
                      
                      <div className="text-sm text-gray-600 space-y-1">
                          <div className="flex items-center gap-2">
                              <i className="fas fa-envelope text-gray-400 w-4"></i> {member.email}
                          </div>
                          <div className="flex items-center gap-2">
                              <i className="fas fa-phone text-gray-400 w-4"></i> {member.phone}
                          </div>
                      </div>

                      {/* Permissions Summary */}
                      <div className="mt-3 pt-3 border-t border-gray-100">
                          <p className="text-xs font-bold text-gray-500 mb-1">الصلاحيات الممنوحة:</p>
                          <div className="flex flex-wrap gap-1">
                              {member.permissions.includes('all') ? (
                                  <span className="text-[10px] bg-blue-100 text-blue-800 px-2 py-0.5 rounded font-bold">صلاحيات كاملة (Admin)</span>
                              ) : member.permissions.length > 0 ? (
                                  member.permissions.slice(0, 3).map(p => (
                                      <span key={p} className="text-[10px] bg-gray-100 text-gray-600 px-2 py-0.5 rounded">
                                          {AVAILABLE_PERMISSIONS.find(ap => ap.id === p)?.label || p}
                                      </span>
                                  ))
                              ) : (
                                  <span className="text-[10px] bg-red-50 text-red-500 px-2 py-0.5 rounded">بلا صلاحيات</span>
                              )}
                              {!member.permissions.includes('all') && member.permissions.length > 3 && (
                                  <span className="text-[10px] text-gray-400">+{member.permissions.length - 3}</span>
                              )}
                          </div>
                      </div>

                      <div className="mt-4 flex justify-end gap-2">
                          <button onClick={() => handleEdit(member)} className="text-gray-400 hover:text-primary text-sm flex items-center gap-1">
                              <i className="fas fa-cog"></i> إدارة الصلاحيات
                          </button>
                          <button onClick={() => handleDelete(member.id)} className="text-red-400 hover:text-red-600 text-sm">
                              <i className="fas fa-trash"></i>
                          </button>
                      </div>
                  </div>
              </div>
          ))}
      </div>

      <Modal isOpen={isModalOpen} onClose={() => setIsModalOpen(false)} title={editingStaff ? "تعديل صلاحيات الموظف" : "إضافة موظف جديد"}>
          <form onSubmit={handleSubmit} className="space-y-4">
              <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">الاسم الكامل</label>
                  <input name="name" type="text" defaultValue={editingStaff?.name} className="w-full px-3 py-2 border rounded-lg bg-white text-gray-900" required />
              </div>
              <div className="grid grid-cols-2 gap-4">
                  <div>
                      <label className="block text-sm font-medium text-gray-700 mb-1">المسمى الوظيفي</label>
                      <select name="role" defaultValue={editingStaff?.role} className="w-full px-3 py-2 border rounded-lg bg-white text-gray-900">
                          <option value="doctor">طبيب (صلاحيات كاملة)</option>
                          <option value="nurse">ممرض/ـة</option>
                          <option value="receptionist">موظف استقبال</option>
                      </select>
                  </div>
                  <div>
                      <label className="block text-sm font-medium text-gray-700 mb-1">الحالة</label>
                      <select name="status" defaultValue={editingStaff?.status || 'active'} className="w-full px-3 py-2 border rounded-lg bg-white text-gray-900">
                          <option value="active">نشط</option>
                          <option value="inactive">غير نشط</option>
                      </select>
                  </div>
              </div>
              
              {/* Login Credentials */}
              <div className="bg-blue-50 p-3 rounded-lg border border-blue-100">
                  <h4 className="text-xs font-bold text-blue-800 mb-2 flex items-center gap-1"><i className="fas fa-key"></i> بيانات الدخول</h4>
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                      <div>
                          <label className="block text-xs font-medium text-gray-700 mb-1">البريد الإلكتروني</label>
                          <input name="email" type="email" defaultValue={editingStaff?.email} className="w-full px-3 py-2 border rounded-lg bg-white text-gray-900" dir="ltr" required />
                      </div>
                      <div>
                          <label className="block text-xs font-medium text-gray-700 mb-1">كلمة المرور</label>
                          <input 
                            name="password" 
                            type="password" 
                            className="w-full px-3 py-2 border rounded-lg bg-white text-gray-900" 
                            dir="ltr" 
                            placeholder={editingStaff ? "اتركها فارغة للإبقاء" : ""}
                            required={!editingStaff}
                          />
                      </div>
                  </div>
              </div>

              <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">رقم الهاتف</label>
                  <input name="phone" type="tel" defaultValue={editingStaff?.phone} className="w-full px-3 py-2 border rounded-lg bg-white text-gray-900" dir="ltr" />
              </div>

              {/* Permission Matrix */}
              <div className="bg-gray-50 p-4 rounded-xl border border-gray-200 mt-2">
                  <h4 className="font-bold text-gray-800 mb-3 text-sm flex items-center gap-2">
                      <i className="fas fa-lock text-primary"></i> تحديد الصلاحيات والخدمات
                  </h4>
                  <div className="grid grid-cols-1 sm:grid-cols-2 gap-3">
                      {AVAILABLE_PERMISSIONS.map(perm => (
                          <label key={perm.id} className="flex items-center gap-2 cursor-pointer p-2 bg-white rounded border border-gray-100 hover:border-blue-300 transition-all select-none">
                              <input 
                                  type="checkbox" 
                                  checked={selectedPermissions.includes(perm.id)}
                                  onChange={() => togglePermission(perm.id)}
                                  className="w-4 h-4 text-primary rounded focus:ring-primary"
                              />
                              <span className="text-sm text-gray-700">{perm.label}</span>
                          </label>
                      ))}
                  </div>
                  <p className="text-xs text-gray-500 mt-3">* ملاحظة: الطبيب يحصل تلقائياً على كافة الصلاحيات.</p>
              </div>

              <button type="submit" className="w-full bg-primary text-white py-3 rounded-xl font-bold hover:bg-primary-dark transition-colors shadow-md mt-2">
                  {editingStaff ? 'حفظ التعديلات' : 'إضافة الموظف'}
              </button>
          </form>
      </Modal>
    </div>
  );
};
