
import React, { useState } from 'react';
import { Modal } from '../../components/Modal';
import { useToast } from '../../context/ToastContext';
import { useData } from '../../context/DataContext';

export const Messages: React.FC = () => {
    const { patients } = useData();
    const { addToast } = useToast();
    const [isModalOpen, setIsModalOpen] = useState(false);
    
    // Mock Messages
    const [messages, setMessages] = useState([
        { id: 1, sender: 'منى علي', text: 'هل يمكن تأجيل الموعد لساعة؟', time: '10:30 AM', read: false },
        { id: 2, sender: 'خالد عمر', text: 'شكراً دكتور، الألم خف كثيراً.', time: 'أمس', read: true },
        { id: 3, sender: 'فاطمة حسن', text: 'متى تظهر نتائج التحليل؟', time: 'أمس', read: true },
        { id: 4, sender: 'سعيد محمد', text: 'أريد حجز موعد جديد للأسبوع القادم.', time: 'منذ يومين', read: true },
        { id: 5, sender: 'ليلى أحمد', text: 'الدواء سبب لي بعض الدوخة، هل هذا طبيعي؟', time: 'منذ يومين', read: false },
    ]);

    const handleMarkRead = (id: number) => {
        setMessages(prev => prev.map(m => m.id === id ? { ...m, read: true } : m));
    };

    const handleSendMessage = (e: React.FormEvent) => {
        e.preventDefault();
        const form = e.target as HTMLFormElement;
        const recipient = (form.elements.namedItem('recipient') as HTMLSelectElement).value;
        const message = (form.elements.namedItem('message') as HTMLTextAreaElement).value;

        if (recipient && message) {
            addToast('تم إرسال الرسالة بنجاح', 'success');
            setIsModalOpen(false);
            // In a real app, this would append to sent messages or start a thread
        }
    };

    return (
        <div className="h-[calc(100vh-8rem)] flex flex-col bg-white rounded-2xl shadow-sm border border-gray-100 overflow-hidden">
            <div className="p-4 border-b border-gray-100 flex justify-between items-center bg-gray-50">
                <h2 className="font-bold text-gray-800 flex items-center gap-2">
                    <i className="fas fa-envelope-open-text text-primary"></i>
                    رسائل المرضى
                </h2>
                <span className="bg-primary text-white text-xs px-2 py-1 rounded-full font-bold">{messages.filter(m => !m.read).length} غير مقروء</span>
            </div>
            <div className="flex-1 overflow-y-auto">
                {messages.map(msg => (
                    <div 
                        key={msg.id} 
                        onClick={() => handleMarkRead(msg.id)}
                        className={`p-4 border-b border-gray-50 hover:bg-gray-50 cursor-pointer transition-colors group ${!msg.read ? 'bg-blue-50/50' : ''}`}
                    >
                        <div className="flex justify-between items-start mb-1">
                            <div className="flex items-center gap-2">
                                <div className={`w-2 h-2 rounded-full ${!msg.read ? 'bg-primary' : 'bg-transparent'}`}></div>
                                <h4 className={`text-sm ${!msg.read ? 'font-bold text-gray-900' : 'font-medium text-gray-700'}`}>{msg.sender}</h4>
                            </div>
                            <span className="text-xs text-gray-400 group-hover:text-primary transition-colors">{msg.time}</span>
                        </div>
                        <p className={`text-sm pr-4 ${!msg.read ? 'text-gray-800 font-medium' : 'text-gray-500'} truncate`}>{msg.text}</p>
                    </div>
                ))}
                {messages.length === 0 && (
                    <div className="text-center py-10 text-gray-400">لا توجد رسائل جديدة</div>
                )}
            </div>
            <div className="p-4 border-t border-gray-100 bg-gray-50">
                <button 
                    onClick={() => setIsModalOpen(true)}
                    className="w-full py-2 bg-white border border-gray-200 rounded-lg text-gray-600 hover:text-primary hover:border-primary transition-colors text-sm font-medium shadow-sm"
                >
                    <i className="fas fa-plus mr-1"></i> رسالة جديدة
                </button>
            </div>

            <Modal isOpen={isModalOpen} onClose={() => setIsModalOpen(false)} title="إرسال رسالة جديدة">
                <form onSubmit={handleSendMessage} className="space-y-4">
                    <div>
                        <label className="block text-sm font-medium text-gray-700 mb-1">المستلم (المريض)</label>
                        <select name="recipient" className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-primary focus:border-primary bg-white text-gray-900" required>
                            <option value="">اختر مريضاً...</option>
                            {patients.map(p => (
                                <option key={p.id} value={p.name}>{p.name}</option>
                            ))}
                        </select>
                    </div>
                    <div>
                        <label className="block text-sm font-medium text-gray-700 mb-1">نص الرسالة</label>
                        <textarea 
                            name="message" 
                            rows={5} 
                            className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-primary focus:border-primary bg-white text-gray-900" 
                            placeholder="اكتب رسالتك هنا..."
                            required
                        ></textarea>
                    </div>
                    <button type="submit" className="w-full bg-primary text-white py-2.5 rounded-xl font-bold hover:bg-primary-dark transition-colors">
                        إرسال
                    </button>
                </form>
            </Modal>
        </div>
    );
};
