
import React, { useState } from 'react';
import { useData } from '../../context/DataContext';
import { SupportTicket } from '../../types';
import { Modal } from '../../components/Modal';
import { useToast } from '../../context/ToastContext';

export const SupportCenter: React.FC = () => {
  const { supportTickets, updateTicketStatus } = useData();
  const { addToast } = useToast();
  const [filter, setFilter] = useState<'all' | 'open' | 'in_progress' | 'resolved'>('all');
  
  // Reply Modal State
  const [replyModalOpen, setReplyModalOpen] = useState(false);
  const [selectedTicket, setSelectedTicket] = useState<SupportTicket | null>(null);
  const [replyText, setReplyText] = useState('');

  const filteredTickets = filter === 'all' 
    ? supportTickets 
    : supportTickets.filter(t => t.status === filter);

  const getPriorityColor = (p: string) => {
      switch(p) {
          case 'high': return 'bg-red-100 text-red-800 border-red-200';
          case 'medium': return 'bg-yellow-100 text-yellow-800 border-yellow-200';
          default: return 'bg-gray-100 text-gray-800 border-gray-200';
      }
  };

  const getStatusColor = (s: string) => {
      switch(s) {
          case 'open': return 'bg-blue-100 text-blue-800';
          case 'in_progress': return 'bg-purple-100 text-purple-800';
          case 'resolved': return 'bg-green-100 text-green-800';
          default: return 'bg-gray-100';
      }
  };

  const getStatusLabel = (s: string) => {
      switch(s) {
          case 'open': return 'مفتوحة';
          case 'in_progress': return 'قيد المعالجة';
          case 'resolved': return 'تم الحل';
          default: return s;
      }
  };

  const handleOpenReply = (ticket: SupportTicket) => {
      setSelectedTicket(ticket);
      setReplyModalOpen(true);
      setReplyText('');
  };

  const sendReply = () => {
      if (selectedTicket && replyText) {
          addToast(`تم إرسال الرد إلى ${selectedTicket.tenantName}`, 'success');
          if (selectedTicket.status === 'open') {
              updateTicketStatus(selectedTicket.id, 'in_progress');
          }
          setReplyModalOpen(false);
          setSelectedTicket(null);
      } else {
          addToast('يرجى كتابة نص الرد', 'error');
      }
  };

  return (
    <div className="space-y-6">
      <div className="flex justify-between items-center mb-6">
        <div>
          <h2 className="text-2xl font-bold text-gray-800">مركز الدعم الفني</h2>
          <p className="text-gray-500">إدارة تذاكر الدعم وطلبات العملاء</p>
        </div>
      </div>

      <div className="flex gap-4 border-b border-gray-200 pb-1">
          {['all', 'open', 'in_progress', 'resolved'].map((f) => (
              <button 
                key={f}
                onClick={() => setFilter(f as any)}
                className={`px-4 py-2 font-medium text-sm border-b-2 transition-colors ${
                    filter === f ? 'border-primary text-primary' : 'border-transparent text-gray-500 hover:text-gray-700'
                }`}
              >
                  {f === 'all' ? 'الكل' : getStatusLabel(f)}
              </button>
          ))}
      </div>

      <div className="grid gap-4">
          {filteredTickets.map(ticket => (
              <div key={ticket.id} className="bg-white p-6 rounded-2xl shadow-sm border border-gray-100 flex flex-col md:flex-row gap-4 items-start md:items-center">
                  <div className="flex-1">
                      <div className="flex items-center gap-3 mb-2">
                          <span className={`text-xs px-2 py-1 rounded-lg border font-bold uppercase ${getPriorityColor(ticket.priority)}`}>
                              {ticket.priority}
                          </span>
                          <span className="text-sm text-gray-500">#{ticket.id}</span>
                          <span className="text-sm text-gray-400">• {ticket.createdAt}</span>
                      </div>
                      <h3 className="font-bold text-lg text-gray-800 mb-1">{ticket.subject}</h3>
                      <p className="text-gray-600 text-sm mb-2">{ticket.message}</p>
                      <div className="text-xs text-primary font-semibold flex items-center gap-1">
                          <i className="fas fa-clinic-medical"></i>
                          {ticket.tenantName}
                      </div>
                  </div>

                  <div className="flex flex-col gap-2 min-w-[150px]">
                      <span className={`px-3 py-1 rounded-full text-center text-xs font-bold ${getStatusColor(ticket.status)}`}>
                          {getStatusLabel(ticket.status)}
                      </span>
                      <div className="flex gap-1 justify-center mt-2">
                          {ticket.status !== 'in_progress' && ticket.status !== 'resolved' && (
                              <button 
                                onClick={() => updateTicketStatus(ticket.id, 'in_progress')}
                                className="p-2 bg-gray-100 rounded hover:bg-purple-100 hover:text-purple-600 transition-colors" title="بدء المعالجة"
                              >
                                  <i className="fas fa-play"></i>
                              </button>
                          )}
                          {ticket.status !== 'resolved' && (
                              <button 
                                onClick={() => updateTicketStatus(ticket.id, 'resolved')}
                                className="p-2 bg-gray-100 rounded hover:bg-green-100 hover:text-green-600 transition-colors" title="إغلاق التذكرة"
                              >
                                  <i className="fas fa-check"></i>
                              </button>
                          )}
                          <button 
                            onClick={() => handleOpenReply(ticket)}
                            className="p-2 bg-gray-100 rounded hover:bg-blue-100 hover:text-blue-600 transition-colors" title="رد"
                          >
                              <i className="fas fa-reply"></i>
                          </button>
                      </div>
                  </div>
              </div>
          ))}
          {filteredTickets.length === 0 && (
              <div className="text-center py-12 text-gray-400">لا توجد تذاكر في هذه القائمة</div>
          )}
      </div>

      <Modal isOpen={replyModalOpen} onClose={() => setReplyModalOpen(false)} title={`الرد على التذكرة #${selectedTicket?.id}`}>
          <div className="space-y-4">
              <div className="bg-gray-50 p-4 rounded-lg text-sm text-gray-700 border border-gray-200">
                  <strong>المشكلة:</strong> {selectedTicket?.message}
              </div>
              <div>
                  <label className="block text-sm font-medium text-gray-700 mb-1">نص الرد</label>
                  <textarea 
                    rows={5} 
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-primary focus:border-primary bg-white text-gray-900"
                    placeholder="اكتب ردك هنا..."
                    value={replyText}
                    onChange={(e) => setReplyText(e.target.value)}
                  ></textarea>
              </div>
              <button 
                onClick={sendReply}
                className="w-full bg-primary text-white py-2.5 rounded-xl font-bold hover:bg-primary-dark transition-colors"
              >
                  إرسال الرد
              </button>
          </div>
      </Modal>
    </div>
  );
};
