
import React, { useState } from 'react';
import { useToast } from '../../context/ToastContext';

export const PregnancyTracker: React.FC = () => {
  const [lmp, setLmp] = useState('');
  const { addToast } = useToast();
  
  const calculatePregnancy = (dateString: string) => {
    if (!dateString) return null;
    const lmpDate = new Date(dateString);
    const today = new Date();
    
    // EDD is LMP + 280 days
    const eddDate = new Date(lmpDate);
    eddDate.setDate(lmpDate.getDate() + 280);
    
    // Gestational Age
    const diffTime = Math.abs(today.getTime() - lmpDate.getTime());
    const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24)); 
    const weeks = Math.floor(diffDays / 7);
    const days = diffDays % 7;
    
    const progress = Math.min((diffDays / 280) * 100, 100);

    return {
      edd: eddDate.toISOString().split('T')[0],
      weeks,
      days,
      progress
    };
  };

  const data = calculatePregnancy(lmp);

  const handleAction = (action: string) => {
      addToast(`تم فتح سجل ${action} بنجاح`, 'success');
  };

  return (
    <div className="bg-white rounded-2xl shadow-sm border border-gray-100 p-6">
      <div className="flex justify-between items-center mb-6 border-b border-gray-100 pb-4">
        <h3 className="text-lg font-bold text-gray-800 flex items-center gap-2">
            <i className="fas fa-baby text-pink-500"></i>
            متابعة الحمل (OB/GYN)
        </h3>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
        <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">تاريخ آخر دورة شهرية (LMP)</label>
            <input 
                type="date" 
                value={lmp}
                onChange={(e) => setLmp(e.target.value)}
                className="w-full px-4 py-2 border border-gray-300 rounded-xl focus:ring-primary focus:border-primary bg-white text-gray-900"
            />
            <p className="text-xs text-gray-500 mt-2">يتم حساب موعد الولادة المتوقع وعمر الحمل تلقائياً.</p>
        </div>

        {data ? (
            <div className="bg-pink-50 rounded-2xl p-5 border border-pink-100">
                <div className="flex justify-between items-end mb-4">
                    <div>
                        <p className="text-sm text-pink-600 font-medium">عمر الحمل الحالي</p>
                        <h4 className="text-3xl font-bold text-gray-800">
                            {data.weeks} <span className="text-base font-normal text-gray-500">أسبوع</span> و {data.days} <span className="text-base font-normal text-gray-500">يوم</span>
                        </h4>
                    </div>
                    <div className="text-left">
                        <p className="text-sm text-pink-600 font-medium">موعد الولادة (EDD)</p>
                        <h4 className="text-xl font-bold text-gray-800">{data.edd}</h4>
                    </div>
                </div>

                {/* Progress Bar */}
                <div className="relative pt-1">
                    <div className="flex mb-2 items-center justify-between">
                        <div className="text-right">
                        <span className="text-xs font-semibold inline-block text-pink-600">
                            الثلث الأول
                        </span>
                        </div>
                         <div className="text-right">
                        <span className="text-xs font-semibold inline-block text-pink-600">
                            الثلث الثاني
                        </span>
                        </div>
                         <div className="text-right">
                        <span className="text-xs font-semibold inline-block text-pink-600">
                            الثلث الثالث
                        </span>
                        </div>
                    </div>
                    <div className="overflow-hidden h-3 mb-4 text-xs flex rounded-full bg-pink-200">
                        <div style={{ width: `${data.progress}%` }} className="shadow-none flex flex-col text-center whitespace-nowrap text-white justify-center bg-pink-500 transition-all duration-1000"></div>
                    </div>
                </div>
            </div>
        ) : (
            <div className="flex items-center justify-center bg-gray-50 rounded-2xl border border-dashed border-gray-300 text-gray-400 text-sm">
                يرجى إدخال تاريخ LMP لعرض البيانات
            </div>
        )}
      </div>
      
      {data && (
        <div className="mt-6 grid grid-cols-2 md:grid-cols-4 gap-4">
            <button onClick={() => handleAction('الوزن والضغط')} className="p-3 rounded-xl border border-gray-200 hover:border-pink-500 hover:bg-pink-50 transition-all text-sm font-medium text-gray-700">
                <i className="fas fa-weight text-pink-500 mb-1 block text-lg"></i>
                الوزن والضغط
            </button>
             <button onClick={() => handleAction('السونار')} className="p-3 rounded-xl border border-gray-200 hover:border-pink-500 hover:bg-pink-50 transition-all text-sm font-medium text-gray-700">
                <i className="fas fa-wave-square text-pink-500 mb-1 block text-lg"></i>
                سونار (Ultrasound)
            </button>
             <button onClick={() => handleAction('التحاليل')} className="p-3 rounded-xl border border-gray-200 hover:border-pink-500 hover:bg-pink-50 transition-all text-sm font-medium text-gray-700">
                <i className="fas fa-vial text-pink-500 mb-1 block text-lg"></i>
                تحاليل دورية
            </button>
             <button onClick={() => handleAction('التاريخ الطبي')} className="p-3 rounded-xl border border-gray-200 hover:border-pink-500 hover:bg-pink-50 transition-all text-sm font-medium text-gray-700">
                <i className="fas fa-history text-pink-500 mb-1 block text-lg"></i>
                سجل الولادات
            </button>
        </div>
      )}
    </div>
  );
};
