
import React, { useState } from 'react';
import { useToast } from '../../context/ToastContext';

export const DentalChart: React.FC = () => {
  // Mock teeth state: 1-32. Status: 'healthy', 'decay', 'treated', 'missing'
  const [teeth, setTeeth] = useState<Record<number, string>>({});
  const { addToast } = useToast();

  const handleToothClick = (id: number) => {
    setTeeth(prev => {
      const current = prev[id] || 'healthy';
      let next = 'healthy';
      if (current === 'healthy') next = 'decay';
      else if (current === 'decay') next = 'treated';
      else if (current === 'treated') next = 'missing';
      else next = 'healthy';
      return { ...prev, [id]: next };
    });
  };

  const getToothColor = (status?: string) => {
    switch (status) {
      case 'decay': return 'text-yellow-500 drop-shadow-sm';
      case 'treated': return 'text-blue-500 drop-shadow-sm';
      case 'missing': return 'text-gray-200 opacity-50';
      default: return 'text-white drop-shadow-md shadow-gray-400 stroke-1 stroke-gray-400'; 
    }
  };

  const renderTeethRow = (start: number, end: number, reverse = false) => {
    let ids = [];
    for (let i = start; i <= end; i++) ids.push(i);
    if (reverse) ids = ids.reverse();

    return (
      <div className="flex justify-center gap-1 sm:gap-2 md:gap-3">
        {ids.map(id => (
          <div key={id} className="flex flex-col items-center cursor-pointer group transition-transform hover:scale-110" onClick={() => handleToothClick(id)}>
            <i 
              className={`fas fa-tooth text-2xl md:text-4xl transition-all duration-300 ${getToothColor(teeth[id])}`}
              style={{ 
                  filter: teeth[id] === 'healthy' || !teeth[id] ? 'drop-shadow(0px 2px 2px rgba(0,0,0,0.1))' : 'none',
                  WebkitTextStroke: '1px #cbd5e1'
              }}
            ></i>
            <span className="text-[10px] text-gray-400 font-bold mt-1 group-hover:text-primary">{id}</span>
          </div>
        ))}
      </div>
    );
  };

  const handleSave = () => {
      addToast('تم تحديث مخطط الأسنان وحفظ الحالة بنجاح', 'success');
  };

  return (
    <div className="bg-white rounded-3xl shadow-sm border border-gray-200 p-6 md:p-8">
      <div className="flex flex-col md:flex-row justify-between items-center mb-8 border-b border-gray-100 pb-6 gap-4">
        <h3 className="text-xl font-bold text-gray-800 flex items-center gap-2">
            <i className="fas fa-teeth text-emerald-500"></i>
            مخطط الأسنان التفاعلي
        </h3>
        <div className="flex flex-wrap justify-center gap-4 text-xs font-bold bg-gray-50 px-4 py-2 rounded-xl">
            <div className="flex items-center gap-2"><div className="w-3 h-3 bg-white border border-gray-400 rounded-full"></div> سليم</div>
            <div className="flex items-center gap-2"><div className="w-3 h-3 bg-yellow-500 rounded-full shadow-sm"></div> تسوس</div>
            <div className="flex items-center gap-2"><div className="w-3 h-3 bg-blue-500 rounded-full shadow-sm"></div> معالج</div>
            <div className="flex items-center gap-2"><div className="w-3 h-3 bg-gray-200 rounded-full border border-dashed border-gray-400"></div> مخلوع</div>
        </div>
      </div>

      <div className="space-y-10 py-6 bg-slate-50/50 rounded-3xl p-4 border border-slate-100 relative overflow-hidden">
        <div className="absolute top-0 left-0 w-full h-2 bg-gradient-to-r from-transparent via-emerald-200 to-transparent opacity-30"></div>
        
        {/* Upper Jaw */}
        <div className="space-y-3">
            <div className="text-center text-xs font-extrabold text-emerald-600/50 uppercase tracking-widest">Upper Jaw (الفك العلوي)</div>
            <div className="flex flex-wrap justify-center gap-8 md:gap-16">
                {renderTeethRow(18, 11, true)} {/* Right Upper */}
                {renderTeethRow(21, 28)}       {/* Left Upper */}
            </div>
        </div>

        <div className="relative">
            <div className="absolute inset-0 flex items-center">
                <div className="w-full border-t border-slate-200 border-dashed"></div>
            </div>
            <div className="relative flex justify-center text-xs uppercase">
                <span className="bg-slate-50 px-2 text-slate-400">Midline</span>
            </div>
        </div>

        {/* Lower Jaw */}
        <div className="space-y-3">
            <div className="flex flex-wrap justify-center gap-8 md:gap-16">
                {renderTeethRow(48, 41, true)} {/* Right Lower */}
                {renderTeethRow(31, 38)}       {/* Left Lower */}
            </div>
            <div className="text-center text-xs font-extrabold text-emerald-600/50 uppercase tracking-widest mt-2">Lower Jaw (الفك السفلي)</div>
        </div>
      </div>
      
      <div className="mt-8 bg-gray-50 rounded-xl p-4 border border-gray-100">
        <h4 className="font-bold text-sm text-gray-800 mb-3 border-b border-gray-200 pb-2">ملخص الحالة:</h4>
        <div className="flex flex-wrap gap-2 mb-4 min-h-[40px]">
            {Object.entries(teeth).map(([id, status]) => (
                status !== 'healthy' && (
                    <span key={id} className="px-3 py-1.5 bg-white border border-gray-200 rounded-lg text-xs font-bold shadow-sm flex items-center gap-2 animate-fade-in">
                        <span className="w-5 h-5 rounded-full bg-gray-100 flex items-center justify-center text-[10px]">{id}</span>
                        <span className={`${status === 'decay' ? 'text-yellow-600' : status === 'treated' ? 'text-blue-600' : 'text-gray-400'}`}>
                            {status === 'decay' ? 'تسوس' : status === 'treated' ? 'معالج' : 'مفقود'}
                        </span>
                    </span>
                )
            ))}
            {Object.keys(teeth).every(k => teeth[k as any] === 'healthy') && (
                <span className="text-gray-400 text-sm italic py-1">لا توجد ملاحظات مسجلة حالياً. اضغط على الأسنان لتغيير الحالة.</span>
            )}
        </div>
        <button 
            onClick={handleSave}
            className="w-full py-3 bg-emerald-600 text-white rounded-xl font-bold hover:bg-emerald-700 transition-colors shadow-md flex items-center justify-center gap-2"
        >
            <i className="fas fa-save"></i> حفظ تخطيط الأسنان
        </button>
      </div>
    </div>
  );
};
